package com.elitescloud.boot.excel.util;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.poi.excel.RowUtil;
import cn.hutool.poi.excel.cell.CellUtil;
import cn.hutool.poi.excel.style.StyleUtil;
import com.alibaba.excel.converters.Converter;
import com.alibaba.excel.write.builder.ExcelWriterBuilder;
import com.elitescloud.boot.common.annotation.BigDecimalFormat;
import com.elitescloud.boot.excel.config.tmpl.converter.ConverterFactory;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.boot.provider.DataFormatterProvider;
import com.elitescloud.boot.util.BeanWrapperUtil;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.ser.std.NumberSerializers;
import lombok.extern.slf4j.Slf4j;
import org.apache.poi.ss.SpreadsheetVersion;
import org.apache.poi.ss.usermodel.Cell;
import org.apache.poi.ss.usermodel.CellType;
import org.apache.poi.ss.usermodel.Sheet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

import java.math.BigDecimal;
import java.util.List;
import java.util.Set;

/**
 * Excel工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/5
 */
@Slf4j
public final class ExcelUtil extends cn.hutool.poi.excel.ExcelUtil {
    private static final Logger LOG = LoggerFactory.getLogger(ExcelUtil.class);

    private static List<Converter<?>> converters;

    private ExcelUtil() {
    }

    /**
     * 复制行
     * <p>
     * 从startIndex复制到endIndex，包含startIndex和endIndex行，索引按从0开始
     *
     * @param source            源文件
     * @param target            目标文件
     * @param startIndex        开始行的索引，以0开始
     * @param endIndex          结束行的索引
     */
    public static void copyRows(@NonNull Sheet source, @NonNull Sheet target, int startIndex, int endIndex) {
        copyRows(source, target, startIndex, endIndex, null);
    }

    /**
     * 复制行
     * <p>
     * 从startIndex复制到endIndex，包含startIndex和endIndex行，索引按从0开始
     *
     * @param source            源文件
     * @param target            目标文件
     * @param startIndex        开始行的索引，以0开始
     * @param endIndex          结束行的索引
     * @param ignoredRowIndexes 忽略的行
     */
    public static void copyRows(@NonNull Sheet source, @NonNull Sheet target, int startIndex, int endIndex, Set<Integer> ignoredRowIndexes) {
        Assert.isTrue(endIndex > startIndex, "endIndex必须大于startIndex");
        var startRowNum = source.getFirstRowNum();
        var endRowNum = source.getLastRowNum();
        if (startIndex > endRowNum || endIndex < startRowNum) {
            // 没有需要复制的行
            LOG.warn("没有可复制的行数：{}-{}, {}-{}", startIndex, endIndex, startRowNum, endRowNum);
            return;
        }

        // 开始复制
        int maxColNum = 0;
        for (int i = startIndex; i <= endIndex; i++) {
            if (ignoredRowIndexes != null && ignoredRowIndexes.contains(i)) {
                continue;
            }
            var sourceRow = source.getRow(i);
            int colNum = sourceRow == null ? -1 : sourceRow.getLastCellNum();
            if (colNum < 1) {
                // 没有有效的数据列
                continue;
            }
            var targetRow = RowUtil.getOrCreateRow(target, i);
            targetRow.setHeightInPoints(sourceRow.getHeightInPoints());
            int numOfEmptyCell = 0;
            for (int j = sourceRow.getFirstCellNum(); j < colNum; j++) {
                if (i == startIndex) {
                    // 设置列宽
                    target.setColumnWidth(j, source.getColumnWidth(j));
                }

                var sourceCell = sourceRow.getCell(j);
                if (sourceCell == null) {
                    if (j >= maxColNum) {
                        maxColNum = j;
                    }
                    break;
                }
                // 判断是否还有列
                var isEmpty = sourceCell.getCellTypeEnum() == CellType.BLANK;
                if (isEmpty) {
                    numOfEmptyCell++;
                    if (j >= maxColNum && numOfEmptyCell >= 5) {
                        maxColNum = j;
                        break;
                    }
                } else {
                    numOfEmptyCell = 0;
                }

                var targetCell = CellUtil.getOrCreateCell(targetRow, j);
                try {
                    copyCell(sourceCell, targetCell);
                } catch (Exception e) {
                    throw new BusinessException("处理数据失败，第" + (i + 1) + "行第" + (j + 1) + "列", e);
                }
            }
        }
    }

    /**
     * 复制单元格
     *
     * @param source 源
     * @param target 目标
     */
    public static void copyCell(@NonNull Cell source, @NonNull Cell target) {
        // 复制样式
        target.setCellStyle(StyleUtil.cloneCellStyle(target, source.getCellStyle()));

        // 复制值
        var cellType = source.getCellTypeEnum();
        switch (cellType) {
            case STRING:
                target.setCellValue(source.getStringCellValue());
                break;
            case NUMERIC:
                target.setCellValue(source.getNumericCellValue());
                break;
            case _NONE:
                break;
            case BLANK:
                break;
            case ERROR:
                target.setCellErrorValue(source.getErrorCellValue());
                break;
            case BOOLEAN:
                target.setCellValue(source.getBooleanCellValue());
                break;
            case FORMULA:
                target.setCellFormula(source.getCellFormula());
                break;
            default:
                target.setCellValue(source.getStringCellValue());
        }
    }

    /**
     * 构建ObjectMapper
     *
     * @return objectMapper
     */
    public static ObjectMapper buildObjectMapper() {
        var objectMapperBuilder = ObjectMapperFactory.builderInstance();

        // 继续还使用转数字，避免导出时，数字类型会转成字符串影响统计
        objectMapperBuilder.serializerByType(Long.class, new NumberSerializers.LongSerializer(Long.class));
        objectMapperBuilder.serializerByType(Long.TYPE, new NumberSerializers.LongSerializer(Long.TYPE));
        return objectMapperBuilder.build();
    }

    /**
     * 格式化excel writer
     *
     * @param writerBuilder excel文件
     * @return excelWriterBuilder
     */
    public static ExcelWriterBuilder normalizeExcelWriter(ExcelWriterBuilder writerBuilder) {
        writerBuilder
                .useDefaultStyle(true)
                .autoTrim(true)
                .writeExcelOnException(true);

        // 设置转换器
        for (Converter<?> converter : getConverters()) {
            writerBuilder.registerConverter(converter);
        }

        return writerBuilder;
    }

    /**
     * 格式化导出的值
     *
     * @param value    待格式化的数据
     * @param field    字段名称
     * @param dataType 数据对象类型
     * @return 值
     */
    public static Object formatExportedValue(Object value, String field, Class<?> dataType) {
        if (value == null || CharSequenceUtil.isBlank(field) || dataType == null) {
            return value;
        }

        if (value instanceof String) {
            // 避免字符串过长，超出单元格的长度
            return CharSequenceUtil.subPre(value.toString(), SpreadsheetVersion.EXCEL2007.getMaxTextLength());
        }
        if (value instanceof BigDecimal) {
            BigDecimalFormat bigDecimalFormat = BeanWrapperUtil.getAnnotation(dataType, field, BigDecimalFormat.class);
            return DataFormatterProvider.getInstance().format((BigDecimal) value, bigDecimalFormat);
        }

        return value;
    }

    /**
     * 获取转换器
     *
     * @return 转换器
     */
    public static List<Converter<?>> getConverters() {
        if (converters == null) {
            converters = ConverterFactory.getConvertersAll();
        }
        return converters;
    }
}
