package com.elitescloud.boot.jpa;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.flyway.FlywayCloudtProperties;
import com.elitescloud.boot.jpa.config.tenant.HibernateTenantDatasourceProvider;
import com.elitescloud.boot.jpa.config.tenant.HibernateTenantIdentifierResolver;
import com.elitescloud.boot.support.AbstractEnvironmentPostProcessor;
import com.elitescloud.boot.support.CloudtStarterTool;
import com.elitescloud.cloudt.common.constant.TenantIsolateStrategy;
import org.hibernate.MultiTenancyStrategy;
import org.springframework.boot.context.event.ApplicationContextInitializedEvent;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.util.Map;

/**
 * 租户相关默认环境变量设置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/3/25
 */
@Configuration(proxyBeanMethods = false)
class CloudtJpaTenantEnvironmentListener implements SmartApplicationListener, Ordered {
    private static final String TENANT_CONFIG_PREFIX = "elitesland.tenant.client";

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
        return eventType.isAssignableFrom(ApplicationContextInitializedEvent.class);
    }

    @Override
    public void onApplicationEvent(@NonNull ApplicationEvent event) {
        if (event instanceof ApplicationContextInitializedEvent) {
            Environment env = ((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment();
            if (!(env instanceof StandardEnvironment)) {
                return;
            }

            StandardEnvironment environment = (StandardEnvironment) ((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment();
            if (!CloudtStarterTool.enabledTenant(environment)) {
                // 未启用租户
                return;
            }

            var propertySource = environment.getPropertySources().get(AbstractEnvironmentPostProcessor.PROPERTY_SOURCE_NAME);
            if (propertySource == null) {
                throw new IllegalStateException("缺少默认PropertySource");
            }
            Map<String, Object> defaultProperties = (Map<String, Object>) propertySource.getSource();

            // 隔离策略
            MultiTenancyStrategy tenancyStrategy = getTenantStrategy(environment);
            if (tenancyStrategy == MultiTenancyStrategy.SCHEMA) {
                // 基于schema隔离
                propertiesForSchemaIsolate(environment, defaultProperties);
            } else if (tenancyStrategy == MultiTenancyStrategy.DATABASE) {
                // 基于数据库隔离
            } else if (tenancyStrategy == MultiTenancyStrategy.DISCRIMINATOR) {
                // 基于字段隔离
                propertiesForFieldIsolate(environment, defaultProperties);
            }
        }
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

    private static void propertiesForFieldIsolate(ConfigurableEnvironment environment, Map<String, Object> defaultProperties) {
        defaultProperties.put("spring.jpa.properties.hibernate.multiTenancy", MultiTenancyStrategy.DISCRIMINATOR.name());
        defaultProperties.put("spring.jpa.properties.hibernate.tenant_identifier_resolver", HibernateTenantIdentifierResolver.class.getName());
    }

    private static void propertiesForSchemaIsolate(ConfigurableEnvironment environment, Map<String, Object> defaultProperties) {
        defaultProperties.put("spring.jpa.properties.hibernate.multiTenancy", MultiTenancyStrategy.SCHEMA.name());
        defaultProperties.put("spring.jpa.properties.hibernate.tenant_identifier_resolver", HibernateTenantIdentifierResolver.class.getName());
        defaultProperties.put("spring.jpa.properties.hibernate.multi_tenant_connection_provider", HibernateTenantDatasourceProvider.class.getName());

        // flyway的默认schema
        String schemaDefault = environment.getProperty(FlywayCloudtProperties.CONFIG_PREFIX + ".schema-default");
        if (StringUtils.hasText(schemaDefault)) {
            schemaDefault = environment.getProperty(TENANT_CONFIG_PREFIX + ".default-schema");
        }
        if (StringUtils.hasText(schemaDefault)) {
            defaultProperties.put(FlywayCloudtProperties.CONFIG_PREFIX + ".schema-default", schemaDefault);
            defaultProperties.put(TENANT_CONFIG_PREFIX + ".default-schema", schemaDefault);
        }

        // 默认开启flyway
        String enabledFlyway = environment.getProperty(FlywayCloudtProperties.CONFIG_PREFIX + ".enabled", "true");
        defaultProperties.put(FlywayCloudtProperties.CONFIG_PREFIX + ".enabled", enabledFlyway);

        // 关闭自动创建表结构
        defaultProperties.put("spring.jpa.hibernate.ddl-auto", "none");
        defaultProperties.put("spring.jpa.properties.hibernate.hbm2ddl.auto", "none");
    }

    /**
     * 获取租户策略
     *
     * @param environment
     * @return
     */
    private static MultiTenancyStrategy getTenantStrategy(Environment environment) {
        String isoLateStrategy = environment.getProperty(TENANT_CONFIG_PREFIX + ".isolate-strategy");
        if (!StringUtils.hasText(isoLateStrategy) || CharSequenceUtil.equals(isoLateStrategy, TenantIsolateStrategy.NONE.name())) {
            // 未设置
            return MultiTenancyStrategy.NONE;
        }

        if (CharSequenceUtil.equals(isoLateStrategy, TenantIsolateStrategy.SCHEMA.name())) {
            return MultiTenancyStrategy.SCHEMA;
        }
        if (CharSequenceUtil.equals(isoLateStrategy, TenantIsolateStrategy.FIELD.name())) {
            return MultiTenancyStrategy.DISCRIMINATOR;
        }
        if (CharSequenceUtil.equals(isoLateStrategy, TenantIsolateStrategy.DATABASE.name())) {
            return MultiTenancyStrategy.DATABASE;
        }

        return MultiTenancyStrategy.NONE;
    }
}
