package com.elitescloud.boot.jpa.config.auditing;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.context.TenantOrgContext;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.authorization.core.SecurityContextUtil;
import com.elitescloud.cloudt.common.constant.CommonConstant;
import com.elitescloud.cloudt.common.constant.TenantConstant;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang3.StringUtils;
import org.springframework.data.auditing.AuditingHandler;
import org.springframework.data.mapping.PersistentEntity;
import org.springframework.data.mapping.PersistentProperty;
import org.springframework.data.mapping.context.MappingContext;
import org.springframework.data.mapping.context.PersistentEntities;

import javax.annotation.Nonnull;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 自定义审计处理器.
 *
 * @author Kaiser（wang shao）
 * @date 2021-03-04
 */
@Slf4j
public class CustomAuditHandler extends AuditingHandler {

    private final boolean defaultSecInitEnable = false;

    private TenantClientProvider tenantClientProvider;
    private List<AuditCustomizer> customizers;

    public CustomAuditHandler(MappingContext<? extends PersistentEntity<?, ?>, ? extends PersistentProperty<?>> mappingContext) {
        super(mappingContext);
    }

    public CustomAuditHandler(PersistentEntities entities) {
        super(entities);
    }

    @Override
    public <T> T markCreated(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // ========= 1. 基本审计字段，默认值 =========
            // 当前时间
            LocalDateTime now = LocalDateTime.now();
            // 当前操作用户ID
            Long currentUserId = 0L;
            // 当前操作用户名称/账号名称
            String currentUsername = null;
            // 数据权限审计字段，归属组织ID
            Long secBuId = null;
            // 数据权限审计字段，归属员工ID
            Long secUseId = null;
            // 租户id,归属于用户信息
            Long tenantId = entity.getTenantId();
            // 租户组织ID
            Long tenantOrgId = entity.getTenantOrgId();
            Long orgId = ObjectUtil.defaultIfNull(entity.getBelongOrgId(), entity.getSecBuId());
            if (tenantId == null) {
                var currentTenant = getTenantClientProvider().getSessionTenant();
                if (currentTenant != null) {
                    tenantId = currentTenant.getId();
                }
            }

            var user = currentUser();

            // 租户组织ID
            if (tenantOrgId == null) {
                tenantOrgId = this.obtainTenantOrgId(user);
            }

            // ========= 2. 初始化部分审计字段 =========
            if (user != null && user.getUser() != null) {
                // *** 当前用户ID ***
                currentUserId = user.getUser().getId();
                // *** 当前用户名称/账号名称 ***
                currentUsername = user.getUsername();

                // 4. 数据权限审计字段，只有绑定了员工信息才会设置，没有绑定则为空
                if (orgId == null) {
                    orgId = user.getOrgId();
                }
            }


            // ========= 3. 设置审计字段 =========
            // 创建信息，创建人ID和name，要么都更新，要么都不更新，保证一致
            if (entity.getCreateUserId() == null && StringUtils.isBlank(entity.getCreator())) {
                entity.setCreateUserId(currentUserId);
                entity.setCreator(currentUsername);
            }
            if (entity.getCreateTime() == null) {
                entity.setCreateTime(now);
            }

            // 修改信息，修改人ID和name，要么都更新，要么都不更新，保证一致
            if (entity.getModifyUserId() == null && StringUtils.isBlank(entity.getUpdater())) {
                entity.setModifyUserId(currentUserId);
                entity.setUpdater(currentUsername);
            }
            if (entity.getModifyTime() == null) {
                entity.setModifyTime(now);
            }

            // 删除标识
            if (entity.getDeleteFlag() == null) {
                entity.setDeleteFlag(CommonConstant.COMMON_DELETE_NO);
            }

            // 数据权限相关信息
            if (defaultSecInitEnable) {
                if (entity.getSecBuId() == null && secBuId != null) {
                    entity.setSecBuId(secBuId);
                }
                if (entity.getSecUserId() == null && secUseId != null) {
                    entity.setSecUserId(secUseId);
                }
            }

            entity.setTenantId(ObjectUtil.defaultIfNull(tenantId, TenantConstant.DEFAULT_TENANT_ID));
            entity.setTenantOrgId(tenantOrgId);
            entity.setBelongOrgId(orgId);
        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onCreate(source));
        return source;
    }

    @Override
    public <T> T markModified(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // ========= 1. 基本审计字段，默认值 =========
            var now = LocalDateTime.now();
            // 当前操作人ID
            Long currentUserId = entity.getModifyUserId();
            // 当前操作人名称/账号名
            String currentUsername = entity.getUpdater();


            // ========= 2. 初始化部分审计字段 =========
            var user = currentUser();
            if (user != null && user.getUser() != null) {
                // *** 当前用户ID ***
                currentUserId = user.getUser().getId();
                // *** 当前用户名称/账号名称 ***
                currentUsername = user.getUsername();
            }


            // ========= 3. 设置审计字段 =========
            entity.setModifyUserId(currentUserId);
            entity.setUpdater(currentUsername);
            entity.setModifyTime(now);
            if (entity.getDeleteFlag() == null) {
                entity.setDeleteFlag(CommonConstant.COMMON_DELETE_NO);
            }

        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onUpdate(source));
        return source;
    }

    private Long obtainTenantOrgId(GeneralUserDetails user) {
        var useTenantOrg = TenantOrgContext.isUseTenantOrg();
        if (!ObjectUtil.defaultIfNull(useTenantOrg, true)) {
            // 不使用租户组织
            return null;
        }

        Long tenantOrgId = TenantOrgContext.getTenantOrg();
        if (tenantOrgId != null) {
            return tenantOrgId;
        }
        if (user != null) {
            return user.getTenantOrgId();
        }

        return null;
    }

    private GeneralUserDetails currentUser() {
        return SecurityContextUtil.currentUser();
    }

    public TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }

    public List<AuditCustomizer> getCustomizers() {
        if (customizers == null) {
            customizers = SpringContextHolder.getObjectProvider(AuditCustomizer.class).stream().collect(Collectors.toList());
        }
        return customizers;
    }
}
