package com.elitescloud.boot.jpa.common;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.text.CharSequenceUtil;
import com.blazebit.persistence.CriteriaBuilderFactory;
import com.blazebit.persistence.PagedList;
import com.blazebit.persistence.querydsl.BlazeJPAQuery;
import com.elitescloud.cloudt.common.base.PagingVO;
import com.elitescloud.cloudt.common.base.QBaseModel;
import com.querydsl.core.types.*;
import com.querydsl.core.types.dsl.*;
import com.querydsl.jpa.impl.JPAQuery;
import com.querydsl.jpa.impl.JPAQueryFactory;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Sort;
import org.springframework.util.StringUtils;

import javax.persistence.EntityManager;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Optional;
import java.util.function.Supplier;

/**
 * 基本的RepoProc类.
 *
 * @author Kaiser（wang shao）
 * @date 2021/12/22
 */
@Log4j2
public abstract class BaseRepoProc<T extends Serializable> {
    protected JPAQueryFactory jpaQueryFactory;
    protected EntityManager entityManager;
    protected CriteriaBuilderFactory criteriaBuilderFactory;

    /**
     * T的Q类实例
     */
    private final EntityPathBase<T> qModel;
    private PathBuilder<?> pathBuilder;

    protected BaseRepoProc(@NotNull EntityPathBase<T> qModel) {
        this.qModel = qModel;
    }

    /**
     * 根据ID删除
     *
     * @param id ID
     * @return 删除结果
     */
    public long delete(long id) {
        return jpaQueryFactory.delete(qModel)
                .where(getIdPath().eq(id))
                .execute();
    }

    /**
     * 根据ID删除
     *
     * @param ids ID
     * @return 删除结果
     */
    public long delete(@NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.delete(qModel)
                .where(getIdPath().in(ids))
                .execute();
    }

    /**
     * 根据某字段删除数据
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段路径
     * @return 删除记录数
     */
    protected <P, Q extends Path<P>> long deleteByValue(@NotNull Q path, P value) {
        return jpaQueryFactory.delete(qModel)
                .where(fieldEq(path, value))
                .execute();
    }

    /**
     * 根据某字段删除数据
     *
     * @param path   字段path
     * @param values 字段值
     * @param <P>    值类型
     * @param <Q>    字段路径
     * @return 删除记录数
     */
    protected <P, Q extends Path<P>> long deleteByValue(@NotNull Q path, @NotEmpty Collection<P> values) {
        return jpaQueryFactory.delete(qModel)
                .where(fieldIn(path, values))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param id 数据记录ID
     * @return 是否更新成功
     */
    public long updateDeleteFlag(long id) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 1)
                .where(getIdPath().eq(id))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param id    数据记录ID
     * @param value 删除标记值
     * @return 是否更新成功
     */
    public long updateDeleteFlag(long id, int value) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), value)
                .where(getIdPath().eq(id))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param ids 数据记录ID
     * @return 是否更新成功
     */
    public long updateDeleteFlag(@NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 1)
                .where(getIdPath().in(ids))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param ids   数据记录ID
     * @param value 删除标记值
     * @return 是否更新成功
     */
    public long updateDeleteFlag(@NotEmpty Collection<Long> ids, int value) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), value)
                .where(getIdPath().in(ids))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段路径
     * @return 更新记录数
     */
    protected <P, Q extends Path<P>> long updateDeleteFlagByValue(@NotNull Q path, P value) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 1)
                .where(fieldEq(path, value))
                .execute();
    }

    /**
     * 修改删除标记
     *
     * @param path   字段path
     * @param values 字段值
     * @param <P>    值类型
     * @param <Q>    字段路径
     * @return 更新记录数
     */
    protected <P, Q extends Path<P>> long updateDeleteFlagByValue(@NotNull Q path, @NotEmpty Collection<P> values) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 1)
                .where(fieldIn(path, values))
                .execute();
    }

    /**
     * 移除删除标记
     *
     * @param id 数据记录ID
     * @return 是否更新成功
     */
    public long removeDeleteFlag(long id) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 0)
                .where(getIdPath().eq(id))
                .execute();
    }

    /**
     * 移除删除标记
     *
     * @param ids 数据记录ID
     * @return 是否更新成功
     */
    public long removeDeleteFlag(@NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 0)
                .where(getIdPath().in(ids))
                .execute();
    }

    /**
     * 移除删除标记
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段路径
     * @return 更新记录数
     */
    protected <P, Q extends Path<P>> long removeDeleteFlagByValue(@NotNull Q path, P value) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 0)
                .where(fieldEq(path, value))
                .execute();
    }

    /**
     * 移除删除标记
     *
     * @param path   字段path
     * @param values 字段值
     * @param <P>    值类型
     * @param <Q>    字段路径
     * @return 更新记录数
     */
    protected <P, Q extends Path<P>> long removeDeleteFlagByValue(@NotNull Q path, @NotEmpty Collection<P> values) {
        return jpaQueryFactory.update(qModel)
                .set(getDeleteFlagPath(), 0)
                .where(fieldIn(path, values))
                .execute();
    }

    /**
     * 根据ID获取数据信息
     *
     * @param id ID
     * @return 单条记录
     */
    public T get(long id) {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(getIdPath().eq(id))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据ID获取数据
     *
     * @param id ID
     * @return 数据
     */
    public Optional<T> getOptional(long id) {
        return Optional.ofNullable(get(id));
    }

    /**
     * 根据ID获取数据信息
     *
     * @param ids ID列表
     * @return 单条记录
     */
    public List<T> get(@NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(getIdPath().in(ids))
                .fetch();
    }

    /**
     * 根据ID获取数据信息
     *
     * @param ids ID列表
     * @return 单条记录
     */
    public List<T> get(@NotEmpty Long[] ids) {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(getIdPath().in(ids))
                .fetch();
    }

    /**
     * 根据某字段获取单条记录
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段路径
     * @return 记录
     */
    protected <P, Q extends Path<P>> T getOneByValue(@NotNull Q path, P value) {
        return getOneOptionalByValue(path, value).orElse(null);
    }

    /**
     * 根据某字段获取单条记录
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段路径
     * @return 记录
     */
    protected <P, Q extends Path<P>> Optional<T> getOneOptionalByValue(@NotNull Q path, P value) {
        return Optional.ofNullable(jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(fieldEq(path, value))
                .limit(1)
                .fetchOne());
    }

    /**
     * 查询数据列表
     *
     * @param predicate 查询条件
     * @return 数据列表
     */
    protected List<T> getList(Predicate predicate) {
        return jpaQueryFactory.selectFrom(qModel)
                .where(predicate)
                .fetch();
    }

    /**
     * 根据字段值获取列表
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段类型
     * @return 记录列表
     */
    protected <P, Q extends Path<P>> List<T> getListByValue(@NotNull Q path, P value) {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(fieldEq(path, value))
                .fetch();
    }

    /**
     * 根据某字段列表获取记录列表
     *
     * @param path  字段path
     * @param value 字段值
     * @param <P>   值类型
     * @param <Q>   字段类型
     * @return 记录列表
     */
    protected <P, Q extends Path<P>> List<T> getListByValue(@NotNull Q path, @NotEmpty Collection<P> value) {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(fieldIn(path, value))
                .fetch();
    }

    /**
     * 根据ID判断是否存在
     *
     * @param id ID
     * @return 是否存在
     */
    public boolean exists(long id) {
        return jpaQueryFactory.select(this.getIdPath())
                .from(qModel)
                .where(getIdPath().eq(id))
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 过滤存在的ID
     *
     * @param ids id集合
     * @return 存在的ID
     */
    public List<Long> exists(Collection<Long> ids) {
        return jpaQueryFactory.select(this.getIdPath())
                .from(qModel)
                .where(getIdPath().in(ids))
                .fetch();
    }

    /**
     * 过滤存在的值
     *
     * @param value 值集合
     * @return 存在的值
     */
    public <P, Q extends SimpleExpression<P>> List<P> exists(@NotNull Q path, @NotEmpty Collection<P> value) {
        return jpaQueryFactory.select(path)
                .from(qModel)
                .where(path.in(value))
                .fetch();
    }

    /**
     * 查询所有数据
     *
     * @return 数据列表
     */
    public List<T> all() {
        return this.getList(null);
    }

    /**
     * 查询所有未删除的数据
     *
     * @return 数据列表
     */
    public List<T> allForUnDeleted() {
        return jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(getDeleteFlagPath().eq(0))
                .fetch()
                ;
    }

    /**
     * 获取所有的ID
     *
     * @return 所有ID
     */
    public List<Long> allId() {
        return jpaQueryFactory.select(getIdPath())
                .from(qModel)
                .fetch();
    }

    /**
     * 获取所有的ID
     *
     * @return 所有ID
     */
    public List<Long> allIdForUnDeleted() {
        return jpaQueryFactory.select(getIdPath())
                .from(qModel)
                .where(getDeleteFlagPath().eq(0))
                .fetch();
    }

    /**
     * 根据ID获取单个字段的值
     *
     * @param path 字段的表达式
     * @param id   ID
     * @param <P>  字段类型
     * @param <Q>  表达式类型
     * @return 字段的值
     */
    protected <P, Q extends Path<P>> P getValue(@NotNull Q path, long id) {
        return jpaQueryFactory.select(path)
                .from(qModel)
                .where(getIdPath().eq(id))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据ID获取单个字段的值
     *
     * @param path 字段的表达式
     * @param ids  ID列表
     * @param <P>  字段类型
     * @param <Q>  表达式类型
     * @return 值列表
     */
    protected <P, Q extends Path<P>> List<P> getValue(@NotNull Q path, @NotEmpty Collection<Long> ids) {
        return jpaQueryFactory.select(path)
                .from(qModel)
                .where(getIdPath().in(ids))
                .fetch();
    }

    /**
     * 根据指定字段获取单个字段的值
     *
     * @param path       字段的表达式
     * @param fieldPath  指定字段的表达式
     * @param fieldValue 指定字段的值
     * @param <P>        字段类型
     * @param <Q>        表达式类型
     * @param <X>        指定字段的值类型
     * @param <Y>        指定字段的表达式类型
     * @return 单个字段的值
     */
    protected <P, Q extends Path<P>, X, Y extends Path<X>> P getValueByValue(@NotNull Q path, @NotNull Y fieldPath, X fieldValue) {
        return jpaQueryFactory.select(path)
                .from(qModel)
                .where(this.fieldEq(fieldPath, fieldValue))
                .limit(1)
                .fetchOne();
    }

    /**
     * 根据指定字段获取单个字段的值
     *
     * @param path        字段的表达式
     * @param fieldPath   指定字段的表达式
     * @param fieldValues 指定字段的值集合
     * @param <P>         字段类型
     * @param <Q>         表达式类型
     * @param <X>         指定字段的值类型
     * @param <Y>         指定字段的表达式类型
     * @return 单个字段的值列表
     */
    protected <P, Q extends Path<P>, X, Y extends Path<X>> List<P> getValueByValue(@NotNull Q path, @NotNull Y fieldPath, @NotEmpty Collection<X> fieldValues) {
        return jpaQueryFactory.select(path)
                .from(qModel)
                .where(this.fieldIn(fieldPath, fieldValues))
                .fetch();
    }

    /**
     * 获取ID
     *
     * @param path  字段
     * @param value 字段的值
     * @param <P>   字段类型
     * @param <Q>   path类型
     * @return ID
     */
    @SuppressWarnings("unchecked")
    protected <P, Q extends Path<P>> Long getIdByValue(@NotNull Q path, P value) {
        return getIdOptionalByValue(path, value).orElse(null);
    }

    /**
     * 获取ID
     *
     * @param path  字段
     * @param value 字段的值
     * @param <P>   字段类型
     * @param <Q>   path类型
     * @return ID
     */
    @SuppressWarnings("unchecked")
    protected <P, Q extends Path<P>> Optional<Long> getIdOptionalByValue(@NotNull Q path, P value) {
        return Optional.ofNullable(jpaQueryFactory.select(getIdPath())
                .from(qModel)
                .where(this.fieldEq(path, value))
                .limit(1)
                .fetchOne());
    }

    /**
     * 获取ID
     *
     * @param path  字段
     * @param value 字段的值
     * @param <P>   字段类型
     * @param <Q>   path类型
     * @return ID
     */
    @SuppressWarnings("unchecked")
    protected <P, Q extends Path<P>> List<Long> getIdsByValue(@NotNull Q path, P value) {
        return jpaQueryFactory.select(getIdPath())
                .from(qModel)
                .where(this.fieldEq(path, value))
                .fetch();
    }

    /**
     * 获取ID
     *
     * @param path  字段
     * @param value 字段的值
     * @param <P>   字段类型
     * @param <Q>   path类型
     * @return ID
     */
    @SuppressWarnings("unchecked")
    protected <P, Q extends Path<P>> List<Long> getIdsByValue(@NotNull Q path, @NotEmpty Collection<P> value) {
        return jpaQueryFactory.select(getIdPath())
                .from(qModel)
                .where(fieldIn(path, value))
                .fetch();
    }

    /**
     * 更新单个字段
     *
     * @param value 更新的值
     * @param path  更新的字段
     * @param id    ID记录
     * @param <P>   值类型
     * @param <Q>   字段
     * @return 更新结果
     */
    protected <P, Q extends Path<P>> boolean updateValue(@NotNull Q path, P value, long id) {
        return jpaQueryFactory.update(qModel)
                .set(path, value)
                .where(getIdPath().eq(id))
                .execute() > 0;
    }

    /**
     * 根据指定字段更新字段
     *
     * @param value      更新的值
     * @param path       更新的字段
     * @param fieldPath  指定字段
     * @param fieldValue 指定字段的值
     * @param <P>        值类型
     * @param <Q>        字段
     * @return 更新结果
     */
    protected <P, Q extends Path<P>, X, Y extends Path<X>> boolean updateValueByValue(@NotNull Q path, P value, @NotNull Y fieldPath, X fieldValue) {
        return jpaQueryFactory.update(qModel)
                .set(path, value)
                .where(this.fieldEq(fieldPath, fieldValue))
                .execute() > 0;
    }

    /**
     * 根据指定字段更新字段
     *
     * @param value      更新的值
     * @param path       更新的字段
     * @param fieldPath  指定字段
     * @param fieldValue 指定字段的值
     * @param <P>        值类型
     * @param <Q>        字段
     * @return 更新结果
     */
    protected <P, Q extends Path<P>, X, Y extends Path<X>> boolean updateValueByValue(@NotNull Q path, P value, @NotNull Y fieldPath, @NotEmpty Collection<X> fieldValue) {
        return jpaQueryFactory.update(qModel)
                .set(path, value)
                .where(this.fieldIn(fieldPath, fieldValue))
                .execute() > 0;
    }

    /**
     * 判断数据是否存在
     *
     * @param value 值
     * @param path  字段表达式
     * @param <Q>   值的类型
     * @param <P>   值的表达式
     * @return 是否存在
     */
    protected <Q, P extends Path<Q>> boolean exists(@NotNull P path, @NotNull Q value) {
        return exists(path, value, null);
    }

    /**
     * 判断数据是否存在
     *
     * @param value          值
     * @param path           字段表达式
     * @param includeDeleted 是否包含已删除的
     * @param <Q>            值的类型
     * @param <P>            值的表达式
     * @return 是否存在
     */
    protected <Q, P extends Path<Q>> boolean exists(@NotNull P path, @NotNull Q value, boolean includeDeleted) {
        return exists(path, value, null, includeDeleted);
    }

    /**
     * 判断数据是否存在
     *
     * @param value     值
     * @param path      字段表达式
     * @param excludeId 排除掉的记录ID
     * @param <Q>       值的类型
     * @param <P>       值的表达式
     * @return 是否存在
     */
    protected <Q, P extends Path<Q>> boolean exists(@NotNull P path, @NotNull Q value, Long excludeId) {
        return exists(path, value, excludeId, true);
    }

    /**
     * 判断数据是否存在
     *
     * @param value          值
     * @param path           字段表达式
     * @param excludeId      排除掉的记录ID
     * @param includeDeleted 是否包含以删除的
     * @param <Q>            值的类型
     * @param <P>            值的表达式
     * @return 是否存在
     */
    @SuppressWarnings("unchecked")
    protected <Q, P extends Path<Q>> boolean exists(@NotNull P path, @NotNull Q value, Long excludeId, boolean includeDeleted) {
        var idPath = getIdPath();

        var condition = fieldEq(path, value);
        if (excludeId != null) {
            // 排除掉指定的记录
            condition = condition.and(idPath.ne(excludeId));
        }
        if (!includeDeleted) {
            // 排除掉删除的
            if (qModel instanceof QBaseModel) {
                condition = condition.and(getDeleteFlagPath().eq(0));
            } else {
                throw new IllegalArgumentException(qModel.getMetadata().getName() + "没有【delete_flag】字段");
            }
        }

        return jpaQueryFactory.select(idPath)
                .from(qModel)
                .where(condition)
                .limit(1)
                .fetchOne() != null;
    }

    /**
     * 分页查询
     * <p>
     * 在原有查询条件上增加了count，相当于查询了2次，性能不太好
     *
     * @param jpaQuery    查询字段、条件等
     * @param pageRequest 分页信息
     * @return 结果数据
     */
    protected PagingVO<T> queryByPageWrap(@NotNull JPAQuery<T> jpaQuery, @NotNull PageRequest pageRequest) {
        long count = jpaQuery.fetchCount();
        if (count == 0) {
            // 没有查询到数据
            return PagingVO.empty();
        }

        // 排序
        var sort = pageRequest.getSort();
        if (!sort.isUnsorted()) {
            var orders = sort.stream()
                    .map(o -> new OrderSpecifier(o.isAscending() ? Order.ASC : Order.DESC, convertFieldExpression(o.getProperty())))
                    .toArray(OrderSpecifier[]::new);
            jpaQuery.orderBy(orders);
        }

        // 分页
        jpaQuery.offset(pageRequest.getOffset()).limit(pageRequest.getPageSize());

        return PagingVO.<T>builder().total(count).records(jpaQuery.fetch()).build();
    }

    /**
     * 分页查询
     * <p>
     * 基于blaze-persistence，有对sql进行优化，性能较好
     *
     * @param jpaQuery    查询字段、条件等
     * @param pageRequest 分页信息
     * @return 结果数据
     */
    protected <U extends Serializable> PagingVO<U> queryByPage(@NotNull JPAQuery<U> jpaQuery, @NotNull PageRequest pageRequest) {
        return this.queryByPage(jpaQuery, pageRequest, null);
    }

    /**
     * 分页查询
     * <p>
     * 基于blaze-persistence，有对sql进行优化，性能较好
     *
     * @param jpaQuery     查询字段、条件等
     * @param pageRequest  分页信息
     * @param defaultOrder 默认排序
     * @return 结果数据
     */
    protected <U extends Serializable> PagingVO<U> queryByPage(@NotNull JPAQuery<U> jpaQuery, @NotNull PageRequest pageRequest, OrderSpecifier<?> defaultOrder) {
        // 转为使用blaze-persistence
        PagedList<U> pagedList = new BlazeJPAQuery<U>(entityManager, jpaQuery.getMetadata(), criteriaBuilderFactory)
                .orderBy(this.obtainOrders(pageRequest, defaultOrder))
                .fetchPage((int) pageRequest.getOffset(), pageRequest.getPageSize());

        return PagingVO.<U>builder().total(pagedList.getTotalSize()).records(pagedList).build();
    }

    /**
     * 分页查询
     *
     * @param condition   查询条件
     * @param pageRequest 分页信息
     * @return 结果数据
     */
    protected PagingVO<T> queryByPage(Predicate condition, @NotNull PageRequest pageRequest) {
        return this.queryByPage(condition, pageRequest, null);
    }

    /**
     * 分页查询
     *
     * @param condition    查询条件
     * @param pageRequest  分页信息
     * @param defaultOrder 默认排序
     * @return 结果数据
     */
    protected PagingVO<T> queryByPage(Predicate condition, @NotNull PageRequest pageRequest, OrderSpecifier<?> defaultOrder) {
        var jpaQuery = jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(condition);
        return this.queryByPage(jpaQuery, pageRequest, defaultOrder);
    }

    /**
     * 构建排序
     *
     * @param pageRequest 分页查询请求
     * @return 排序
     */
    protected OrderSpecifier<?>[] obtainOrders(@NotNull PageRequest pageRequest) {
        return this.obtainOrders(pageRequest, null);
    }

    /**
     * 构建排序
     *
     * @param pageRequest  分页查询请求
     * @param defaultOrder 默认排序
     * @return 排序
     */
    protected OrderSpecifier<?>[] obtainOrders(@NotNull PageRequest pageRequest, OrderSpecifier<?> defaultOrder) {
        var sort = pageRequest.getSort();
        defaultOrder = this.normalizeDefaultOrder(defaultOrder);
        if (sort.isUnsorted()) {
            // 默认排序
            return defaultOrder == null ? new OrderSpecifier[]{new OrderSpecifier<>(Order.DESC, getIdPath())} :
                    new OrderSpecifier[]{defaultOrder, new OrderSpecifier<>(Order.DESC, getIdPath())};
        }

        sort = sort.and(Sort.by(Sort.Direction.DESC, "id"));
        return sort.stream()
                .map(o -> new OrderSpecifier(o.isAscending() ? Order.ASC : Order.DESC, convertFieldExpression(o.getProperty())))
                .toArray(OrderSpecifier[]::new);
    }

    /**
     * 构建分页对象
     *
     * @param page     当前页, 基于0
     * @param pageSize 页大小
     * @return 分页对象
     */
    protected PageRequest ofPage(Integer page, Integer pageSize) {
        return ofPage(page, pageSize, Sort.unsorted());
    }

    /**
     * 构建分页对象
     *
     * @param page     当前页, 基于0
     * @param pageSize 页大小
     * @param sort     排序
     * @return 分页对象
     */
    protected PageRequest ofPage(Integer page, Integer pageSize, Sort sort) {
        if (page == null || page < 0) {
            page = 0;
        }
        if (pageSize == null || pageSize < 0) {
            pageSize = 10;
        }
        if (sort == null) {
            sort = Sort.unsorted();
        }
        return PageRequest.of(page, pageSize, sort);
    }

    /**
     * 追加条件
     *
     * @param oldPredicate 原条件
     * @param newPredicate 新条件
     * @return 条件
     */
    protected Predicate andPredicate(Predicate oldPredicate, Predicate newPredicate) {
        if (newPredicate == null) {
            return oldPredicate;
        }
        return ExpressionUtils.allOf(oldPredicate, newPredicate);
    }

    protected NumberPath<Long> getIdPath() {
        return Expressions.numberPath(Long.class, qModel, "id");
    }

    protected NumberPath<Integer> getDeleteFlagPath() {
        return Expressions.numberPath(Integer.class, qModel, "deleteFlag");
    }

    private OrderSpecifier<?> normalizeDefaultOrder(OrderSpecifier<?> defaultOrder) {
        if (defaultOrder == null) {
            return null;
        }
        if ("id".equals(((Path<?>) defaultOrder.getTarget()).getMetadata().getElement())) {
            // 已经是ID了
            return null;
        }
        return defaultOrder;
    }

    private <P, Q extends Path<P>> BooleanExpression fieldEq(@NotNull Q path, P value) {
        var p = (SimpleExpression<P>) path;
        if (value == null) {
            return p.isNull();
        }
        return p.eq(value);
    }

    private <P, Q extends Path<P>> BooleanExpression fieldIn(@NotNull Q path, Collection<P> values) {
        var p = (SimpleExpression<P>) path;

        return p.in(values);
    }

    private Expression<?> convertFieldExpression(String field) {
        if (pathBuilder == null) {
            pathBuilder = new PathBuilder<>(qModel.getClass(), qModel.getMetadata());
        }
        return pathBuilder.get(field);
    }

    public static class PredicateBuilder {
        private final List<Predicate> conditions = new ArrayList<>();

        private PredicateBuilder() {
        }

        public static PredicateBuilder builder() {
            return new PredicateBuilder();
        }

        public PredicateBuilder and(boolean condition, @NotNull Supplier<Predicate> predicate) {
            if (condition) {
                conditions.add(predicate.get());
            }
            return this;
        }

        public <T> PredicateBuilder andEq(@NotNull SimpleExpression<T> expression, T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.eq(value));
                }
                return this;
            }
            if (value != null) {
                conditions.add(expression.eq(value));
                return this;
            }
            return this;
        }

        public <T> PredicateBuilder andEq(boolean condition, @NotNull SimpleExpression<T> expression, T value) {
            if (condition) {
                conditions.add(value == null ? expression.isNull() : expression.eq(value));
            }
            return this;
        }

        public <T> PredicateBuilder andEq(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull SubQueryExpression<T> subQuery) {
            if (condition) {
                conditions.add(expression.eq(subQuery));
            }
            return this;
        }

        public <T> PredicateBuilder andNe(@NotNull SimpleExpression<T> expression, T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.ne(value));
                }
                return this;
            }
            if (value != null) {
                conditions.add(expression.ne(value));
                return this;
            }
            return this;
        }

        public <T> PredicateBuilder andNe(boolean condition, @NotNull SimpleExpression<T> expression, T value) {
            if (condition) {
                conditions.add(value == null ? expression.isNotNull() : expression.ne(value));
            }
            return this;
        }

        public <T> PredicateBuilder andNe(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull SubQueryExpression<T> subQuery) {
            if (condition) {
                conditions.add(expression.ne(subQuery));
            }
            return this;
        }

        public <T> PredicateBuilder andNe(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull Supplier<SubQueryExpression<T>> subQuery) {
            return this.andNe(condition, expression, subQuery.get());
        }

        public <T extends Comparable<?>> PredicateBuilder andLt(@NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.lt(value));
                }
                return this;
            }
            return this.andLt(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andLt(boolean condition, @NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.lt(value));
            }
            return this;
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andLt(@NotNull NumberExpression<T> expression, @NotNull T value) {
            return this.andLt(value != null, expression, value);
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andLt(boolean condition, @NotNull NumberExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.lt(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andLoe(@NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.loe(value));
                }
                return this;
            }
            return this.andLoe(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andLoe(boolean condition, @NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.loe(value));
            }
            return this;
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andLoe(@NotNull NumberExpression<T> expression, @NotNull T value) {
            return this.andLoe(value != null, expression, value);
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andLoe(boolean condition, @NotNull NumberExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.loe(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andGt(@NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.gt(value));
                }
                return this;
            }
            return this.andGt(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andGt(boolean condition, @NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.gt(value));
            }
            return this;
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andGt(@NotNull NumberExpression<T> expression, @NotNull T value) {
            return this.andGt(value != null, expression, value);
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andGt(boolean condition, @NotNull NumberExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.gt(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andGoe(@NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (value instanceof String) {
                if (StringUtils.hasText((String) value)) {
                    conditions.add(expression.goe(value));
                }
                return this;
            }
            return this.andGoe(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andGoe(boolean condition, @NotNull ComparableExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.goe(value));
            }
            return this;
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andGoe(@NotNull NumberExpression<T> expression, @NotNull T value) {
            return this.andGoe(value != null, expression, value);
        }

        public <T extends Number & Comparable<?>> PredicateBuilder andGoe(boolean condition, @NotNull NumberExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.goe(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andAfter(@NotNull TemporalExpression<T> expression, @NotNull T value) {
            return this.andAfter(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andAfter(boolean condition, @NotNull TemporalExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.after(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andBefore(@NotNull TemporalExpression<T> expression, @NotNull T value) {
            return this.andBefore(value != null, expression, value);
        }

        public <T extends Comparable<?>> PredicateBuilder andBefore(boolean condition, @NotNull TemporalExpression<T> expression, @NotNull T value) {
            if (condition) {
                conditions.add(expression.before(value));
            }
            return this;
        }

        public <T extends Comparable<?>> PredicateBuilder andBetween(boolean condition, @NotNull TemporalExpression<T> expression, T start, T end) {
            if (condition) {
                if (start != null && end != null) {
                    conditions.add(expression.between(start, end));
                    return this;
                }
                if (start != null) {
                    conditions.add(expression.after(start));
                }
                if (end != null) {
                    conditions.add(expression.before(end));
                }
            }
            return this;
        }

        public <T> PredicateBuilder andNotNull(boolean condition, @NotNull SimpleExpression<T> expression) {
            if (condition) {
                conditions.add(expression.isNotNull());
            }
            return this;
        }

        public <T> PredicateBuilder andIsNull(boolean condition, @NotNull SimpleExpression<T> expression) {
            if (condition) {
                conditions.add(expression.isNull());
            }
            return this;
        }

        public PredicateBuilder andLike(@NotNull StringExpression expression, @NotBlank String value) {
            return this.andLike(StringUtils.hasText(value), expression, value);
        }

        public PredicateBuilder andLike(boolean condition, @NotNull StringExpression expression, @NotNull String value) {
            if (condition) {
                conditions.add(expression.like("%" + value + "%"));
            }
            return this;
        }

        public PredicateBuilder andLike(@NotNull StringExpression[] expressions, @NotNull String value) {
            return this.andLike(StringUtils.hasText(value), expressions, value);
        }

        public PredicateBuilder andLike(boolean condition, @NotNull StringExpression[] expressions, @NotNull String value) {
            if (condition) {
                String keyword = "%" + value + "%";
                BooleanExpression exp = null;
                for (StringExpression expression : expressions) {
                    if (expression == null) {
                        continue;
                    }
                    exp = exp == null ? expression.like(keyword) : exp.or(expression.like(keyword));
                }
                conditions.add(exp);
            }
            return this;
        }

        public PredicateBuilder andLeftLike(@NotNull StringExpression expression, @NotBlank String value) {
            return this.andLeftLike(StringUtils.hasText(value), expression, value);
        }

        public PredicateBuilder andLeftLike(boolean condition, @NotNull StringExpression expression, @NotBlank String value) {
            if (condition) {
                conditions.add(expression.like("%" + value));
            }
            return this;
        }

        public PredicateBuilder andLeftLike(@NotNull StringExpression[] expressions, @NotBlank String value) {
            return this.andLeftLike(StringUtils.hasText(value), expressions, value);
        }

        public PredicateBuilder andLeftLike(boolean condition, @NotNull StringExpression[] expressions, @NotBlank String value) {
            if (condition) {
                String keyword = "%" + value;
                BooleanExpression exp = null;
                for (StringExpression expression : expressions) {
                    if (expression == null) {
                        continue;
                    }
                    exp = exp == null ? expression.like(keyword) : exp.or(expression.like(keyword));
                }
                conditions.add(exp);
            }
            return this;
        }

        public PredicateBuilder andRightLike(@NotNull StringExpression expression, @NotBlank String value) {
            return this.andRightLike(StringUtils.hasText(value), expression, value);
        }

        public PredicateBuilder andRightLike(boolean condition, @NotNull StringExpression expression, @NotBlank String value) {
            if (condition) {
                conditions.add(expression.like(value + "%"));
            }
            return this;
        }

        public PredicateBuilder andRightLike(@NotNull StringExpression[] expressions, @NotBlank String value) {
            return this.andRightLike(StringUtils.hasText(value), expressions, value);
        }

        public PredicateBuilder andRightLike(boolean condition, @NotNull StringExpression[] expressions, @NotBlank String value) {
            if (condition) {
                String keyword = value + "%";
                BooleanExpression exp = null;
                for (StringExpression expression : expressions) {
                    if (expression == null) {
                        continue;
                    }
                    exp = exp == null ? expression.like(keyword) : exp.or(expression.like(keyword));
                }
                conditions.add(exp);
            }
            return this;
        }

        public <T> PredicateBuilder andIn(@NotNull SimpleExpression<T> expression, @NotEmpty Collection<T> value) {
            return this.andIn(CollUtil.isNotEmpty(value), expression, value);
        }

        public <T> PredicateBuilder andIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotEmpty Collection<T> value) {
            if (condition) {
                conditions.add(expression.in(value));
            }
            return this;
        }

        public <T> PredicateBuilder andIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull SubQueryExpression<T> value) {
            if (condition) {
                conditions.add(expression.in(value));
            }
            return this;
        }

        public <T> PredicateBuilder andIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull Supplier<SubQueryExpression<T>> value) {
            return this.andIn(condition, expression, value.get());
        }

        public <T> PredicateBuilder andNotIn(@NotNull SimpleExpression<T> expression, @NotEmpty Collection<T> value) {
            return this.andNotIn(CollUtil.isNotEmpty(value), expression, value);
        }

        public <T> PredicateBuilder andNotIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotEmpty Collection<T> value) {
            if (condition) {
                conditions.add(expression.notIn(value));
            }
            return this;
        }

        public <T> PredicateBuilder andNotIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull SubQueryExpression<T> value) {
            if (condition) {
                conditions.add(expression.notIn(value));
            }
            return this;
        }

        public <T> PredicateBuilder andNotIn(boolean condition, @NotNull SimpleExpression<T> expression, @NotNull Supplier<SubQueryExpression<T>> value) {
            return this.andNotIn(condition, expression, value.get());
        }

        public PredicateBuilder andOnNotBlank(String field, @NotNull Supplier<Predicate> predicate) {
            if (CharSequenceUtil.isNotBlank(field)) {
                conditions.add(predicate.get());
            }
            return this;
        }

        public PredicateBuilder andOnNotNull(Object field, @NotNull Supplier<Predicate> predicate) {
            if (field != null) {
                conditions.add(predicate.get());
            }
            return this;
        }

        public PredicateBuilder andOnTrue(Boolean field, @NotNull Supplier<Predicate> predicate) {
            if (field != null && field) {
                conditions.add(predicate.get());
            }
            return this;
        }

        public PredicateBuilder andOnFalse(Boolean field, @NotNull Supplier<Predicate> predicate) {
            if (field != null && !field) {
                conditions.add(predicate.get());
            }
            return this;
        }

        /**
         * 所有条件是and关系
         *
         * @return 条件
         */
        public Predicate build() {
            return ExpressionUtils.allOf(conditions);
        }

        /**
         * 所有条件是or关系
         *
         * @return 条件
         */
        public Predicate buildOr() {
            return ExpressionUtils.anyOf(conditions);
        }
    }

    @Autowired
    public void setJpaQueryFactory(JPAQueryFactory jpaQueryFactory) {
        this.jpaQueryFactory = jpaQueryFactory;
    }

    @Autowired
    public void setEntityManager(EntityManager entityManager) {
        this.entityManager = entityManager;
    }

    @Autowired
    public void setCriteriaBuilderFactory(CriteriaBuilderFactory criteriaBuilderFactory) {
        this.criteriaBuilderFactory = criteriaBuilderFactory;
    }
}
