package com.elitescloud.boot.jpa.config;

import com.blazebit.persistence.Criteria;
import com.blazebit.persistence.CriteriaBuilderFactory;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.jpa.CloudtJpaProperties;
import com.elitescloud.boot.model.entity.BaseModel;
import com.querydsl.core.types.EntityPath;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import org.hibernate.cfg.AvailableSettings;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.boot.autoconfigure.orm.jpa.HibernatePropertiesCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Scope;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import java.time.LocalDateTime;

/**
 * @author Michael Li
 * @date 2020-09-19
 */
public class HibernateConfig {

    private final CloudtJpaProperties jpaProperties;

    public HibernateConfig(CloudtJpaProperties jpaProperties) {
        this.jpaProperties = jpaProperties;
    }

    @Bean
    public HibernatePropertiesCustomizer hibernatePropertiesCustomizer() {
        return hibernateProperties -> {
            // sql前的注释 log4jdbc会因为SQL前的注释中的占位符导致打印SQL参数出现错位
            hibernateProperties.put(AvailableSettings.USE_SQL_COMMENTS, false);
//            hibernateProperties.put("hibernate.integrator_provider",
//                    (IntegratorProvider) () -> Collections.singletonList(CommentIntegrator.INSTANCE));
//            hibernateProperties.put(AvailableSettings.STATEMENT_INSPECTOR, HibernateLogInterceptor.class.getName());
        };
    }

    @Bean
    public JPAQueryFactory jpaQueryFactory(EntityManager em) {
        return new CloudtJPAQueryFactory(em);
    }

    @Bean
    @Scope(ConfigurableBeanFactory.SCOPE_SINGLETON)
    public CriteriaBuilderFactory criteriaBuilderFactory(EntityManagerFactory entityManagerFactory) {
        var criteria = Criteria.getDefault();
        criteria.setProperty("com.blazebit.persistence.size_to_count_transformation", jpaProperties.getBlaze().getSizeToCountTransformation().toString());
        criteria.setProperty("com.blazebit.persistence.expression_optimization", jpaProperties.getBlaze().getExpressionOptimization().toString());
        criteria.setProperty("com.blazebit.persistence.inline_id_query", jpaProperties.getBlaze().getInlineIdQuery().toString());
        criteria.setProperty("com.blazebit.persistence.inline_count_query", jpaProperties.getBlaze().getInlineCountQuery().toString());
        criteria.setProperty("com.blazebit.persistence.query_plan_cache_enabled", jpaProperties.getBlaze().getQueryPlanCacheEnabled().toString());
        criteria.setProperty("com.blazebit.persistence.criteria_negation_wrapper", jpaProperties.getBlaze().getCriteriaNegationWrapper().toString());
        return criteria.createCriteriaBuilderFactory(entityManagerFactory);
    }

    static class CloudtJPAQueryFactory extends JPAQueryFactory {

        public CloudtJPAQueryFactory(EntityManager entityManager) {
            super(entityManager);
        }

        @Override
        public JPAUpdateClause update(EntityPath<?> path) {
            var clause = super.update(path);

            // 增加审计
            this.auditForUpdate(path, clause);
            return clause;
        }

        private void auditForUpdate(EntityPath<?> path, JPAUpdateClause clause) {
            if (BaseModel.class.isAssignableFrom(path.getType())) {
                var now = LocalDateTime.now();
                clause.set(Expressions.dateTimePath(LocalDateTime.class, path, "modifyTime"), now);

                // 当前用户
                var user = SecurityContextUtil.currentUser();
                if (user != null) {
                    clause.set(Expressions.numberPath(Long.class, path, "modifyUserId"), user.getUserId());
                    clause.set(Expressions.stringPath(path, "updater"), user.getUsername());
                }
            }
        }
    }
}
