package com.elitescloud.boot.jpa.support.id.provider.uidgenerator.config;

import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.BitsAllocator;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.UidGenerator;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.UidProvider;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.impl.CachedUidGenerator;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker.DatabaseWorkerIdAssigner;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker.ManualWorkerIdAssigner;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker.RedisWorkerIdAssigner;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker.WorkerIdAssigner;
import com.elitescloud.boot.jpa.support.id.util.IdTool;
import com.elitescloud.cloudt.context.util.DatetimeUtil;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.JdbcProperties;
import org.springframework.context.EnvironmentAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.env.Environment;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;

/**
 * uid-generator算法配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/9
 */
@ConditionalOnProperty(prefix = IdProperties.CONFIG_PREFIX, name = "gen-type", havingValue = "UID_GENERATOR")
@Log4j2
public class UidConfig implements EnvironmentAware, InitializingBean {
    @Value("${spring.application.name:#{'unknown'}}")
    private String dataCenterName;
    private final IdProperties idProperties;
    private final DataSource dataSource;
    private Environment environment;

    public UidConfig(IdProperties idProperties, DataSource dataSource) {
        this.idProperties = idProperties;
        this.dataSource = dataSource;
    }

    @Override
    public void setEnvironment(@NonNull Environment environment) {
        this.environment = environment;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        // 打印生成器配置信息
        logGeneratorInfo();
    }

    @Bean
    public UidProvider uidProvider(UidGenerator uidGenerator,
                                   WorkerIdAssigner workerIdAssigner) {
        return new UidProvider(uidGenerator, workerIdAssigner);
    }

    @Bean
    public CachedUidGenerator cachedUidGenerator(WorkerIdAssigner workerIdAssigner) {
        CachedUidGenerator cachedUidGenerator = new CachedUidGenerator();
        var properties = idProperties.getUid();

        var total = properties.getTimeBits() + properties.getWorkerBits() + properties.getSeqBits();
        Assert.isTrue(total == BitsAllocator.TOTAL_BITS - 1, "uid参数配置有误，timeBits + workerBits + seqBits = " + BitsAllocator.TOTAL_BITS);
        cachedUidGenerator.setTimeBits(properties.getTimeBits());
        cachedUidGenerator.setWorkerBits(properties.getWorkerBits());
        cachedUidGenerator.setSeqBits(properties.getSeqBits());

        cachedUidGenerator.setBoostPower(properties.getBoostPower());
        cachedUidGenerator.setPaddingFactor(properties.getPaddingFactor());

        cachedUidGenerator.setEpochStr(properties.getEpochStr());

        cachedUidGenerator.setWorkerIdAssigner(workerIdAssigner);

        return cachedUidGenerator;
    }

    @Bean
    @ConditionalOnProperty(prefix = IdProperties.CONFIG_PREFIX, name = "assigner-type", havingValue = "MANUAL", matchIfMissing = true)
    public WorkerIdAssigner workerIdAssignerManual() {
        return new ManualWorkerIdAssigner(idProperties);
    }


    @Bean
    @ConditionalOnProperty(prefix = IdProperties.CONFIG_PREFIX, name = "assigner-type", havingValue = "DATABASE")
    public WorkerIdAssigner workerIdAssignerDatabase(JdbcProperties properties) {
        var jdbcTemplate = IdTool.buildJdbcTemplate(environment, dataSource, idProperties, properties);
        return new DatabaseWorkerIdAssigner(idProperties, jdbcTemplate, dataCenterName);
    }

    private void logGeneratorInfo() {
        var formatter = DatetimeUtil.FORMATTER_DATE;
        var prop = idProperties.getUid();
        LocalDateTime epochStart = LocalDateTime.of(LocalDate.parse(prop.getEpochStr(), formatter), LocalTime.MIN);
        LocalDateTime epochEnd = epochStart.plus(Duration.ofSeconds(1L << prop.getTimeBits()));

        // 可使用年数
        long years = Duration.between(epochStart, epochEnd).toDays() / 365;

        // 并发数
        long qps = 1L << prop.getSeqBits();
        qps = qps << prop.getBoostPower();

        // 机器数量
        long workers = 1L << prop.getWorkerBits();

        log.info("ID生成器【uid-generator】准备中，可使用年限{}年，从{}至{}，并发量支持{}万每秒，支持的服务数量{}万", years, epochStart.format(formatter), epochEnd.format(formatter),
                qps / 10000.0, workers / 10000.0);
    }

    @Configuration
    @ConditionalOnBean(RedisTemplate.class)
    static class RedisSupportConfig {
        @Value("${spring.application.name:#{'unknown'}}")
        private String dataCenterName;
        private final IdProperties idProperties;

        public RedisSupportConfig(IdProperties idProperties) {
            this.idProperties = idProperties;
        }

        @Bean
        @ConditionalOnProperty(prefix = IdProperties.CONFIG_PREFIX, name = "assigner-type", havingValue = "REDIS")
        public WorkerIdAssigner workerIdAssignerRedis(RedisTemplate redisTemplate) {
            return new RedisWorkerIdAssigner(idProperties, redisTemplate, dataCenterName);
        }
    }
}
