package com.elitescloud.boot.jpa.util;

import org.hibernate.boot.MetadataBuilder;
import org.hibernate.boot.MetadataSources;
import org.hibernate.boot.model.naming.CamelCaseToUnderscoresNamingStrategy;
import org.hibernate.boot.registry.BootstrapServiceRegistry;
import org.hibernate.boot.registry.BootstrapServiceRegistryBuilder;
import org.hibernate.boot.registry.StandardServiceRegistry;
import org.hibernate.boot.registry.StandardServiceRegistryBuilder;
import org.hibernate.cfg.AvailableSettings;
import org.hibernate.dialect.Dialect;
import org.hibernate.dialect.MySQL57Dialect;
import org.hibernate.tool.hbm2ddl.SchemaExport;
import org.hibernate.tool.schema.TargetType;
import org.springframework.beans.factory.annotation.AnnotatedBeanDefinition;
import org.springframework.beans.factory.config.BeanDefinition;
import org.springframework.boot.orm.jpa.hibernate.SpringImplicitNamingStrategy;
import org.springframework.context.annotation.ClassPathScanningCandidateComponentProvider;
import org.springframework.core.type.filter.AnnotationTypeFilter;
import org.springframework.util.StringUtils;

import javax.persistence.Entity;
import javax.persistence.Table;
import java.io.File;
import java.util.EnumSet;
import java.util.HashSet;
import java.util.Set;
import java.util.stream.Collectors;

/**
 * 数据库导出工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/27
 */
public class DataBaseExportUtil {

    /**
     * 实体类所在包路径
     */
    private final Set<String> entityPackages = new HashSet<>(4);

    /**
     * 是否格式化SQL
     */
    private boolean format = true;

    /**
     * 导出的数据库方言
     */
    private Class<? extends Dialect> dialect = MySQL57Dialect.class;

    /**
     * 导出脚本的文件
     */
    private File outputFile;

    /**
     * entity类
     */
    private final Set<String> entityClassNames = new HashSet<>(128);

    protected DataBaseExportUtil() {
    }

    /**
     * 获取工具类实例
     *
     * @return 工具类
     */
    public static DataBaseExportUtil instance() {
        return new DataBaseExportUtil();
    }

    /**
     * 实体类所在包路径
     *
     * @param entityPackage 实体类所在包路径
     * @return 工具类
     */
    public DataBaseExportUtil entityPackage(String entityPackage) {
        if (StringUtils.hasText(entityPackage)) {
            this.entityPackages.add(entityPackage);
        }
        return this;
    }

    /**
     * 是否格式化输出SQL
     *
     * @param format 是否格式化
     * @return 工具类
     */
    public DataBaseExportUtil format(boolean format) {
        this.format = format;
        return this;
    }

    /**
     * 输出ddl的方言
     *
     * @param dialect 输出方言
     * @return 工具类
     */
    public DataBaseExportUtil dialect(Class<? extends Dialect> dialect) {
        this.dialect = dialect;
        return this;
    }

    /**
     * 输出脚本的文件
     *
     * @param outputFile 输出方言
     * @return 工具类
     */
    public DataBaseExportUtil outputFile(File outputFile) {
        this.outputFile = outputFile;
        return this;
    }

    /**
     * entity实体类
     *
     * @param clazz 实体类
     * @return 实体类
     */
    public DataBaseExportUtil entityClass(Class<?>... clazz) {
        for (Class<?> cla : clazz) {
            entityClassNames.add(cla.getName());
        }
        return this;
    }

    /**
     * 导出DDL
     */
    public void exportDdl() {
        // 获取所有的entity
        scanEntity();
        if (entityClassNames.isEmpty()) {
            return;
        }

        // 开始导出
        execExport();
    }

    private void execExport() {
        BootstrapServiceRegistry bootstrapServiceRegistry = new BootstrapServiceRegistryBuilder().build();
        StandardServiceRegistryBuilder ssrBuilder = new StandardServiceRegistryBuilder(bootstrapServiceRegistry)
                .applySetting(AvailableSettings.DIALECT, dialect.getName())
                .applySetting(AvailableSettings.IMPLICIT_NAMING_STRATEGY, SpringImplicitNamingStrategy.class.getName())
                .applySetting(AvailableSettings.PHYSICAL_NAMING_STRATEGY, CamelCaseToUnderscoresNamingStrategy.class.getName());
        StandardServiceRegistry serviceRegistry = ssrBuilder.build();

        MetadataSources metadataSources = new MetadataSources(serviceRegistry);
        for (String entityClassName : entityClassNames) {
            metadataSources.addAnnotatedClassName(entityClassName);
        }
        MetadataBuilder metadataBuilder = metadataSources.getMetadataBuilder();
        var metadata = metadataBuilder.build();

        SchemaExport export = new SchemaExport();
        // 格式化SQL
        export.setFormat(this.format);
        export.setOverrideOutputFileContent();

        EnumSet<TargetType> targetTypes = EnumSet.of(TargetType.STDOUT);
        if (outputFile != null) {
            export.setOutputFile(outputFile.getAbsolutePath());
            targetTypes.add(TargetType.SCRIPT);
        }

        export.execute(targetTypes, SchemaExport.Action.BOTH, metadata);
    }

    private void scanEntity() {
        var componentProvider = new CustomClassPathScanningCandidateComponentProvider();
        componentProvider.addIncludeFilter(new AnnotationTypeFilter(Entity.class));
        componentProvider.addIncludeFilter(new AnnotationTypeFilter(Table.class));

        for (String basePackage : entityPackages) {
            var beanDefinitions = componentProvider.findCandidateComponents(basePackage.replace(".", "/"));
            entityClassNames.addAll(beanDefinitions.stream().map(BeanDefinition::getBeanClassName).collect(Collectors.toList()));
        }
    }

    static class CustomClassPathScanningCandidateComponentProvider extends ClassPathScanningCandidateComponentProvider {
        public CustomClassPathScanningCandidateComponentProvider() {
            super(false);
        }

        @Override
        protected boolean isCandidateComponent(AnnotatedBeanDefinition beanDefinition) {
            return true;
        }
    }

}
