package com.elitescloud.boot.jpa;

import cn.hutool.core.util.RuntimeUtil;
import com.elitescloud.boot.jpa.support.id.common.WorkerAssignerType;
import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.config.SnowflakeConfig;
import com.elitescloud.boot.jpa.support.id.provider.uidgenerator.config.UidConfig;
import com.elitescloud.boot.provider.CloudtIdCreator;
import com.elitescloud.cloudt.core.provider.IdGeneratorProvider;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;
import org.springframework.util.Assert;

/**
 * Id生成器自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/8
 */
@Configuration(proxyBeanMethods = false)
@EnableConfigurationProperties(IdProperties.class)
@Import({SnowflakeConfig.class, UidConfig.class, CloudtIdGeneratorAutoConfiguration.CloudtWorkerConfig.class})
@Log4j2
class CloudtIdGeneratorAutoConfiguration {

    public CloudtIdGeneratorAutoConfiguration(IdProperties idProperties) {
        log.info("ID生成器类型：{}", idProperties.getGenType());
    }

    /**
     * id生成器
     *
     * @return
     */
    @Bean
    public IdGeneratorProvider idGeneratorProvider() {
        return new IdGeneratorProvider();
    }

    static class CloudtWorkerConfig implements SchedulingConfigurer, InitializingBean {
        private final IdProperties idProperties;
        private CloudtIdCreator cloudtIdGenerator;

        public CloudtWorkerConfig(IdProperties idProperties) {
            this.idProperties = idProperties;
        }

        @Override
        public void afterPropertiesSet() throws Exception {
            RuntimeUtil.addShutdownHook(cloudtIdGenerator::destroy);
        }

        @Override
        public void configureTasks(ScheduledTaskRegistrar taskRegistrar) {
            if (idProperties.getAssignerType() == WorkerAssignerType.MANUAL) {
                // 手工指定，无需更新
                return;
            }

            Assert.isTrue(idProperties.getAlive() != null && idProperties.getAlive().toSeconds() > 0,
                    "alive设置有误");
            Assert.isTrue(idProperties.getHeartBeat() != null
                    && idProperties.getHeartBeat().toSeconds() < idProperties.getAlive().toSeconds(), "heartBeat设置有误");

            taskRegistrar.addFixedDelayTask(() -> {
                try {
                    cloudtIdGenerator.refreshWorkerAssigner();
                } catch (Exception e) {
                    log.error("刷新WorkerAssigner失败：", e);
                }
            }, idProperties.getHeartBeat().toMillis());
        }

        @Autowired
        public void setCloudtIdGenerator(CloudtIdCreator cloudtIdGenerator) {
            this.cloudtIdGenerator = cloudtIdGenerator;
        }
    }
}
