package com.elitescloud.boot.jpa.support.id.util;

import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import com.zaxxer.hikari.HikariDataSource;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.autoconfigure.jdbc.JdbcProperties;
import org.springframework.boot.context.properties.bind.Binder;
import org.springframework.core.env.Environment;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.lang.NonNull;
import org.springframework.util.Assert;

import javax.sql.DataSource;
import java.util.Properties;

/**
 * ID工具.
 *
 * @author Kaiser（wang shao）
 * @date 3/17/2023
 */
public class IdTool {

    private IdTool() {
    }

    public static JdbcTemplate buildJdbcTemplate(@NonNull Environment environment, DataSource defaultDatasource,
                                                 @NonNull IdProperties idProperties, @NonNull JdbcProperties properties) {
        JdbcTemplate jdbcTemplate = new JdbcTemplate(loadDatasource(environment, defaultDatasource, idProperties));
        JdbcProperties.Template template = properties.getTemplate();
        jdbcTemplate.setFetchSize(template.getFetchSize());
        jdbcTemplate.setMaxRows(template.getMaxRows());
        if (template.getQueryTimeout() != null) {
            jdbcTemplate.setQueryTimeout((int) template.getQueryTimeout().getSeconds());
        }
        return jdbcTemplate;
    }

    private static DataSource loadDatasource(Environment environment, DataSource defaultDatasource, IdProperties idProperties) {
        if (idProperties.getUseDefaultDatasource()) {
            // 默认数据源
            Assert.state(defaultDatasource != null, "默认数据源未获取到");
            return defaultDatasource;
        }

        var binder = Binder.get(environment);

        var bindResult = binder.bind(IdProperties.CONFIG_PREFIX + ".datasource", DataSourceProperties.class);
        if (!bindResult.isBound()) {
            throw new IllegalStateException("IdGenerator数据源加载失败，请确认已配置");
        }
        var properties = bindResult.get();
        var datasource = properties.initializeDataSourceBuilder().type(HikariDataSource.class).build();
        // hikari配置
        var hikariProperties = hikariProperties(binder, IdProperties.CONFIG_PREFIX + ".datasource", null);
        if (hikariProperties != null) {
            datasource.setDataSourceProperties(hikariProperties);
        }
        datasource.setPoolName("cloudt-id-worker");

        return datasource;
    }

    private static Properties hikariProperties(Binder binder, String prefix, Properties defaultProperties) {
        String name = prefix + ".hikari";
        var bindResult = binder.bind(name, Properties.class);
        if (bindResult.isBound()) {
            Properties properties = bindResult.get();
            if (defaultProperties != null) {
                defaultProperties.forEach(properties::putIfAbsent);
            }
            return properties;
        }
        return defaultProperties;
    }
}
