package com.elitescloud.boot.jpa.support.id.provider.snowflake.assigner;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.util.IdUtil;
import com.elitescloud.boot.jpa.support.id.config.GenType;
import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.WorkerInfo;
import com.elitescloud.boot.jpa.support.id.provider.snowflake.micro.Snowflake;
import lombok.extern.log4j.Log4j2;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;

import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.List;

/**
 * 基于数据库的ID生成器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/7
 */
@Log4j2
public class DatabaseSnowflakeWorkerAssigner extends AbstractSnowflakeWorkerAssigner {
    private static final String SQL_INSERT = "insert into sys_id_generator(id, gen_type, data_center_name, data_center_id, worker_id, instance_ip, create_time, modify_time) value (" +
            "?, ?, ?, ?, ?, ?, ?, ?)";
    private static final String SQL_QUERY_DATA_CENTER_ID = "select data_center_id from sys_id_generator where data_center_name = ? order by create_time asc";
    private static final String SQL_REFRESH = "update sys_id_generator set modify_time = ? where data_center_id = ? and worker_id = ?";
    private static final String SQL_DELETE_EXPIRE = "delete from sys_id_generator where modify_time < ?";
    private static final String SQL_DESTROY = "delete from sys_id_generator where data_center_id = ? and worker_id = ?";

    private final String dataCenterName;
    private final IdProperties idProperties;
    private final JdbcTemplate jdbcTemplate;

    private Long dataCenterId;
    private Long workerId;

    public DatabaseSnowflakeWorkerAssigner(String dataCenterName, IdProperties idProperties, JdbcTemplate jdbcTemplate) {
        this.dataCenterName = dataCenterName;
        this.idProperties = idProperties;
        this.jdbcTemplate = jdbcTemplate;
    }

    @Override
    protected WorkerInfo distribute() {
        // 先生成dataCenterId
        generateDataCenterId();

        // 再生成workerId
        generateWorkerId();

        return new WorkerInfo(dataCenterId, workerId);
    }

    @Override
    public void refreshAlive() {
        LocalDateTime now = LocalDateTime.now();
        try {
            jdbcTemplate.batchUpdate(SQL_REFRESH, Arrays.asList(new Object[]{now, dataCenterId, workerId}, new Object[]{now, dataCenterId, -1}));

            // 删除过期的
            jdbcTemplate.update(SQL_DELETE_EXPIRE, now.minusMinutes(idProperties.getAlive().toMinutes()));
        } catch (DataAccessException e) {
            log.error("刷新IdGenerator WorkerId异常：", e);
        }
    }

    @Override
    public void destroy() {
        try {
            jdbcTemplate.update(SQL_DESTROY, dataCenterId, workerId);
        } catch (Exception ignored) {
        }
    }

    private void generateDataCenterId() {
        // 查询是否有已存在的
        dataCenterId = existsDataCenterId();
        if (dataCenterId != null) {
            return;
        }

        // 不存在则创建
        String ip = super.getIp();
        long max = Snowflake.getMaxDataCenterId();
        long temp = 0;
        while (temp < max) {
            var now = LocalDateTime.now();
            try {
                jdbcTemplate.update(SQL_INSERT, IdUtil.fastSimpleUUID(), GenType.SNOWFLAKE.name(), dataCenterName, temp, -1,
                        ip, now, now);

                // 判断是否有其它实例已创建
                dataCenterId = temp;
                return;
            } catch (DataAccessException e) {
                dataCenterId = existsDataCenterId();
                if (dataCenterId != null) {
                    return;
                }
                temp++;
            }
        }

        throw new IllegalStateException("dataCenterId超过限制" + max);
    }

    private void generateWorkerId() {
        long max = Snowflake.getMaxWorkerId();
        long temp = 0;
        String ip = super.getIp();
        while (temp < max) {
            var now = LocalDateTime.now();
            try {
                jdbcTemplate.update(SQL_INSERT, IdUtil.fastSimpleUUID(), GenType.SNOWFLAKE.name(), dataCenterName, dataCenterId, temp,
                        ip, now, now);
                workerId = temp;
                return;
            } catch (DataAccessException e) {
                temp++;
            }
        }

        throw new IllegalStateException("workerId超过限制" + max);
    }

    private Long existsDataCenterId() {
        List<Long> dataCenterIdList = jdbcTemplate.queryForList(SQL_QUERY_DATA_CENTER_ID, Long.class, dataCenterName);
        if (CollUtil.isNotEmpty(dataCenterIdList)) {
            return dataCenterIdList.get(0);
        }
        return null;
    }
}
