package com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker;

import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import lombok.extern.log4j.Log4j2;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.util.StringUtils;

/**
 * 基于redis的workerId分配.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/9
 */
@Log4j2
public class RedisWorkerIdAssigner extends AbstractWorkerIdAssigner {
    private static String CACHE_KEY = "cloudt_uid";
    private static final int RETRY_FACTORY = 100;

    private final IdProperties idProperties;
    private final RedisTemplate redisTemplate;
    private final String dataCenterName;

    private long workerId;


    public RedisWorkerIdAssigner(IdProperties idProperties, RedisTemplate redisTemplate, String dataCenterName) {
        this.idProperties = idProperties;
        this.redisTemplate = redisTemplate;
        this.dataCenterName = dataCenterName;

        if (StringUtils.hasText(idProperties.getUid().getCachePrefix())) {
            CACHE_KEY = idProperties.getUid().getCachePrefix();
        }
    }

    @Override
    public void refreshAlive() {
        redisTemplate.opsForValue().set(cacheKeyWorkerId(workerId), name(), idProperties.getAlive());
    }

    @Override
    public void destroy() {
        var key = cacheKeyWorkerId(workerId);
        try {
            // 删除当前的workerId
            redisTemplate.delete(key);
        } catch (Exception e) {
            log.error("销毁UID的workerId实例异常：", e);
        }
        log.info("UID的workerId实例销毁：{}", key);
    }

    @Override
    public long assignWorkerId() {
        var id = generateWorkerId();
        this.workerId = id;
        return id;
    }

    private long generateWorkerId() {
        // 最大取值
        long max = super.maxWorkerId(idProperties);
        String value = name();

        int retryTimes = 0;
        while (true) {
            // 随机生成值然后尝试
            long temp = RandomUtil.randomLong(max);
            boolean result = redisTemplate.opsForValue().setIfAbsent(cacheKeyWorkerId(temp), value, idProperties.getAlive());
            if (result) {
                return temp;
            }
            retryTimes++;

            if (retryTimes >= RETRY_FACTORY) {
                log.warn("尝试使用WorkerId第{}次", retryTimes);
            }
        }
    }

    private String cacheKeyWorkerId(Long theWorkerId) {
        return CACHE_KEY + ":" + "worker" + "_" + theWorkerId;
    }

    private String name() {
        return dataCenterName + "::" + super.getIp();
    }
}
