package com.elitescloud.boot.jpa.common;

import cn.hutool.core.util.ObjectUtil;
import com.blazebit.persistence.querydsl.BlazeJPAQuery;
import com.elitescloud.boot.model.bo.TreeNodeBO;
import com.elitescloud.boot.model.entity.BaseTreeModel;
import com.querydsl.core.Tuple;
import com.querydsl.core.types.Path;
import com.querydsl.core.types.Predicate;
import com.querydsl.core.types.Projections;
import com.querydsl.core.types.QBean;
import com.querydsl.core.types.dsl.BooleanExpression;
import com.querydsl.core.types.dsl.EntityPathBase;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.core.types.dsl.NumberPath;
import com.querydsl.jpa.JPAExpressions;
import lombok.extern.log4j.Log4j2;
import org.springframework.util.Assert;

import javax.validation.constraints.NotNull;
import java.util.*;
import java.util.concurrent.ConcurrentHashMap;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 树形model的RepoProc类.
 * <p>
 * BaseTreeModel里的信息(除了sortNo外)不可手动维护<p>
 * 在新增业务数据后保存树节点；<p>
 * 如果需要修改节点信息，需要在保存业务数据前修改；<p>
 * 删除节点信息时需要先删除节点，后删除业务数据；<p>
 *
 * @author Kaiser（wang shao）
 * @date 2022/9/29
 */
@Log4j2
public abstract class BaseTreeRepoProc<T extends BaseTreeModel> extends BaseRepoProc<T> {

    protected final EntityPathBase<T> qModel;

    public BaseTreeRepoProc(@NotNull EntityPathBase<T> qModel) {
        super(qModel);
        this.qModel = qModel;
    }

    /**
     * 保存树节点
     *
     * @param node     节点
     * @param parentId 目的父节点的ID
     * @param sortNo   节点顺序
     */
    public void saveTreeNode(@NotNull T node, Long parentId, Integer sortNo) {
        parentId = ObjectUtil.defaultIfNull(parentId, BaseTreeModel.DEFAULT_PARENT);

        // 寻找左节点
        var leftNodeId = this.findLeftNodeId(parentId, sortNo);
        // 保存
        this.saveTreeNode(node, parentId, leftNodeId);
    }

    /**
     * 保存树节点
     *
     * @param node       节点
     * @param parentId   目的父节点的ID
     * @param leftNodeId 目的左节点的ID
     */
    public void saveTreeNode(@NotNull T node, Long parentId, Long leftNodeId) {
        Assert.notNull(node.getId(), "请先保存节点信息");
        parentId = ObjectUtil.defaultIfNull(parentId, BaseTreeModel.DEFAULT_PARENT);

        if (node.getLft() == null) {
            // 新增的节点
            addNode(node, parentId, leftNodeId);
            return;
        }
        if (parentId == BaseTreeModel.DEFAULT_PARENT) {
            // 根节点时，无需移动
            return;
        }

        // 更新节点
        if (leftNodeId != null && node.getLft().longValue() == leftNodeId) {
            // 左节点没变
            return;
        }
        if (leftNodeId == null && parentId.longValue() == node.getPId()) {
            // 左节点和父节点都未改变，无需移动
            return;
        }
        moveNode(node, parentId, leftNodeId);
    }

    /**
     * 移动树节点
     *
     * @param node         节点
     * @param parentNodeId 目的父节点ID
     * @param leftNodeId   目的左节点ID
     */
    public void moveNode(@NotNull T node, @NotNull Long parentNodeId, Long leftNodeId) {
        TreeNodeBO parentNode = null, leftNode = null;
        if (leftNodeId != null) {
            leftNode = getNodeInfo(leftNodeId);
            Assert.notNull(leftNode, "左节点不存在");
        } else if (parentNodeId != BaseTreeModel.DEFAULT_PARENT) {
            leftNode = getLastChildNode(parentNodeId);
            if (leftNode == null) {
                parentNode = getNodeInfo(parentNodeId);
                Assert.notNull(parentNode, "父节点不存在");
            }
        }

        moveNode(node, parentNode, leftNode);
    }

    /**
     * 是否存在子节点
     *
     * @param node 节点信息
     */
    public boolean existsChildNode(@NotNull T node) {
        if (node.getLft() == null) {
            // 非树节点
            return false;
        }
        return node.getRgt() - node.getLft() > 1;
    }

    /**
     * 判断是否为对应节点的子节点
     *
     * @param nodeId   节点ID
     * @param parentId 父节点ID
     * @return nodeId是否为parentId的子节点
     */
    public boolean isChildNode(@NotNull Long nodeId, @NotNull Long parentId) {
        if (nodeId == null || parentId == null || Objects.equals(nodeId, parentId)) {
            return false;
        }

        var nodeInfoMap = getNodeInfo(Set.of(nodeId, parentId));
        var node = nodeInfoMap.get(nodeId);
        if (node == null || node.getLft() == null) {
            return false;
        }
        var parentNode = nodeInfoMap.get(parentId);
        if (parentNode == null || parentNode.getLft() == null) {
            return false;
        }

        if (!Objects.equals(node.getRootId(), parentNode.getRootId())) {
            // 不在一棵树上
            return false;
        }
        return node.getLft() > parentNode.getLft() && node.getRgt() < parentNode.getRgt();
    }

    /**
     * 是否为子节点或兄弟节点
     *
     * @param nodeId   子节点
     * @param parentId 父节点
     * @return nodeId是否为parentId的兄弟节点或子节点
     */
    public boolean isChildOrBrotherNode(@NotNull Long nodeId, @NotNull Long parentId) {
        if (nodeId == null || parentId == null || Objects.equals(nodeId, parentId)) {
            return false;
        }

        var nodeInfoMap = getNodeInfo(Set.of(nodeId, parentId));
        var node = nodeInfoMap.get(nodeId);
        if (node == null || node.getLft() == null) {
            throw new IllegalArgumentException("当前节点非树节点");
        }
        var parentNode = nodeInfoMap.get(parentId);
        if (parentNode == null || parentNode.getLft() == null) {
            throw new IllegalArgumentException("父节点非树节点");
        }

        if (!Objects.equals(node.getRootId(), parentNode.getRootId())) {
            // 不在一棵树上
            return false;
        }
        if (Objects.equals(node.getPid(), parentNode.getPid())) {
            // 兄弟节点
            return true;
        }
        return node.getLft() > parentNode.getLft() && node.getRgt() < parentNode.getRgt();
    }

    /**
     * 判断是否为对应节点的父节点
     *
     * @param nodeId   节点ID
     * @param parentId 父节点ID
     * @return parentId是否为nodeId的父节点
     */
    public boolean isParentNode(@NotNull Long nodeId, @NotNull Long parentId) {
        return this.isChildNode(nodeId, parentId);
    }

    /**
     * 判断是否为对应节点的父节点
     *
     * @param nodeId   节点ID
     * @param parentId 父节点ID
     * @return parentId是否为nodeId的父节点
     */
    public boolean isParentOrBrotherNode(@NotNull Long nodeId, @NotNull Long parentId) {
        return this.isChildOrBrotherNode(nodeId, parentId);
    }

    /**
     * 寻找左节点
     * <p>
     * 主要用于保存前寻找左节点时
     *
     * @param parentId 父节点
     * @param sortNo   顺序
     * @return 节点ID
     */
    public Long findLeftNodeId(Long parentId, Integer sortNo) {
        parentId = ObjectUtil.defaultIfNull(parentId, BaseTreeModel.DEFAULT_PARENT);

        var predicate = PredicateBuilder.builder()
                .andEq(true, this.getPidPath(), parentId)
                .and(sortNo != null, () -> this.getSortNoPath().lt(sortNo))
                .build();

        return jpaQueryFactory.select(this.getIdPath())
                .from(qModel)
                .where(predicate)
                .orderBy(this.getLftPath().desc())
                .limit(1)
                .fetchOne();
    }

    /**
     * 寻找最大的序号
     * <p>
     * 主要用于保存时，如果前端没有传递顺序号，则默认递增序号。
     *
     * @param parentId 父节点
     * @return 最大序号
     */
    public Integer findMaxSortNo(Long parentId) {
        parentId = ObjectUtil.defaultIfNull(parentId, BaseTreeModel.DEFAULT_PARENT);

        var value = super.jpaQueryFactory.select(this.getSortNoPath().max())
                .from(qModel)
                .where(this.getPidPath().eq(parentId))
                .fetchOne();
        return ObjectUtil.defaultIfNull(value, 0);
    }

    /**
     * 移除树节点
     *
     * @param node 节点
     */
    public void removeTreeNode(@NotNull T node) {
        if (node.getLft() == null) {
            return;
        }
        var pathLeft = getLftPath();
        var pathRight = getRgtPath();
        var pathRoot = getRootPath();

        jpaQueryFactory.update(qModel)
                .set(pathLeft, pathLeft.subtract(2))
                .where(pathLeft.gt(node.getLft()).and(pathRoot.eq(node.getRootId())))
                .execute();
        jpaQueryFactory.update(qModel)
                .set(pathRight, pathRight.subtract(2))
                .where(pathRight.gt(node.getRgt()).and(pathRoot.eq(node.getRootId())))
                .execute();

        node.setLft(null);
        node.setRgt(null);
        node.setPId(null);
        node.setRootId(null);
        node.setDepth(null);

        super.save(node);
    }

    /**
     * 获取树
     *
     * @param pId           父节点ID
     * @param predicate     查询条件
     * @param nodePredicate 过滤器
     * @param nodeMapper    节点数据转换
     * @param <R>
     * @return 树
     */
    @SuppressWarnings("unchecked")
    public <R> List<R> getTree(Long pId, Predicate predicate, java.util.function.Predicate<T> nodePredicate, Function<T, R> nodeMapper) {
        // 查询节点数据
        Predicate condition = null;
        if (pId != null) {
            var nodeInfo = getNodeInfo(pId);
            Assert.notNull(nodeInfo, "节点[" + pId + "]不存在");
            condition = getRootPath().eq(nodeInfo.getRootId())
                    .and(getLftPath().goe(nodeInfo.getLft()))
                    .and(getRgtPath().loe(nodeInfo.getRgt()));
        }
        if (predicate != null) {
            condition = andPredicate(condition, predicate);
        }
        var treeNodeList = jpaQueryFactory.select(qModel)
                .from(qModel)
                .where(condition)
                .fetch();
        if (treeNodeList.isEmpty()) {
            return (List<R>) treeNodeList;
        }

        // 过滤数据
        treeNodeList = filterTreeNode(treeNodeList, nodePredicate);
        if (nodeMapper == null) {
            return (List<R>) treeNodeList;
        }

        // 数据转换
        return treeNodeList
                .stream()
                .map(nodeMapper)
                .collect(Collectors.toList());
    }

    /**
     * 重构树
     * <p>
     * 重新构建树的左右节点
     *
     * @param nodeId      节点id，如果传递，则重构对应的树，不传递，则重构所有树
     * @param getParentId 获取父节点id
     */
    public void rebuildTree(Long nodeId, @NotNull Function<T, Long> getParentId) {
        // 先清空左右节点
        var rootId = this.clearNodeInfo(nodeId);
        log.info("重构树：{}", rootId);

        // 查询出满足条件的节点
        var dataMaps = jpaQueryFactory.select(qModel).from(qModel)
                .where(rootId == null ? null : getRootPath().eq(rootId))
                .fetch()
                .stream()
                .collect(Collectors.groupingBy(t -> {
                    var pId = getParentId.apply(t);
                    return ObjectUtil.defaultIfNull(pId, BaseTreeModel.DEFAULT_PARENT);
                }));
        if (dataMaps.isEmpty()) {
            return;
        }

        // 获取根
        var roots = dataMaps.getOrDefault(BaseTreeModel.DEFAULT_PARENT, Collections.emptyList());
        Assert.notEmpty(roots, "未查询到根节点数据");

        // 添加数据
        this.addNodeWithChildren(roots, BaseTreeModel.DEFAULT_PARENT, dataMaps);
    }

    /**
     * 查询直接下级ID
     *
     * @param pid 父ID
     * @return 直接下级ID
     */
    public List<Long> getIdByPid(long pid) {
        return super.jpaQueryFactory.select(this.getIdPath())
                .from(qModel)
                .where(this.getPidPath().eq(pid))
                .fetch();
    }

    /**
     * 查询所有下级ID
     *
     * @param pid 父ID
     * @return 下级ID
     */
    public List<Long> getChildrenIdByPid(long pid) {
        var nodeInfo = getNodeInfo(pid);
        if (nodeInfo == null || nodeInfo.getLft() == null) {
            return Collections.emptyList();
        }
        return super.jpaQueryFactory.select(this.getIdPath())
                .from(qModel)
                .where(this.getLftPath().gt(nodeInfo.getLft()).and(this.getRgtPath().lt(nodeInfo.getRgt())).and(this.getRootPath().eq(nodeInfo.getRootId())))
                .fetch();
    }

    /**
     * 获取祖先节点的查询条件
     *
     * @param id 节点ID
     * @return 查询条件
     */
    protected BooleanExpression predicateForParents(@NotNull Long id) {
        var nodeInfo = getNodeInfo(id);
        Assert.notNull(nodeInfo, "节点[" + id + "]不存在");

        return getRootPath().eq(nodeInfo.getRootId())
                .and(getLftPath().loe(nodeInfo.getLft()))
                .and(getRgtPath().goe(nodeInfo.getRgt()));
    }

    /**
     * 获取子孙节点的查询条件
     *
     * @param id 节点ID
     * @return 查询条件
     */
    protected BooleanExpression predicateForChildren(@NotNull Long id) {
        var nodeInfo = getNodeInfo(id);
        Assert.notNull(nodeInfo, "节点[" + id + "]不存在");

        return getRootPath().eq(nodeInfo.getRootId())
                .and(getLftPath().goe(nodeInfo.getLft()))
                .and(getRgtPath().loe(nodeInfo.getRgt()));
    }

    /**
     * 是否有子节点
     *
     * @param lfg 左节点
     * @param rgt 右节点
     * @return 是否有子节点
     */
    protected boolean hasChildren(Integer lfg, Integer rgt) {
        if (lfg == null || rgt == null) {
            return false;
        }
        return rgt - lfg > 1;
    }

    /**
     * 过滤查询上级ID
     *
     * @param parentModel     父model
     * @param parentPredicate 上级过滤条件
     * @param predicate       过滤条件
     * @return id与上级ID的map映射
     */
    protected Map<Long, Long> filterParentId(@NotNull EntityPathBase<T> parentModel, Predicate parentPredicate, Predicate predicate) {
        var selectParentId = JPAExpressions.select(Expressions.numberPath(Long.class, parentModel, PROPERTY_ID))
                .from(parentModel)
                .where(Expressions.numberPath(Long.class, parentModel, PROPERTY_ROOT).eq(getRootPath())
                        .and(Expressions.numberPath(Long.class, parentModel, PROPERTY_LFT).loe(getLftPath()))
                        .and(Expressions.numberPath(Long.class, parentModel, PROPERTY_RGT).goe(getRgtPath()))
                        .and(parentPredicate))
                .orderBy(Expressions.numberPath(Long.class, parentModel, PROPERTY_LFT).desc())
                .limit(1);

        var jpaQuery = jpaQueryFactory.select(getIdPath(), selectParentId)
                .from(qModel)
                .where(predicate);
        var blazeJpaQuery = new BlazeJPAQuery<Tuple>(this.entityManager, jpaQuery.getMetadata(), this.criteriaBuilderFactory);
        return blazeJpaQuery.fetch().stream()
                .collect(Collectors.toMap(t -> t.get(0, Long.class), t -> ObjectUtil.defaultIfNull(t.get(1, Long.class), -1L), (t1, t2) -> t1));
    }

    private List<T> filterTreeNode(List<T> nodeList, java.util.function.Predicate<T> nodePredicate) {
        // 需要过滤掉的节点
        List<T> filterNodes = new ArrayList<>(nodeList.size());
        List<T> candidateNodes, excludeNodes;
        while (true) {
            candidateNodes = new ArrayList<>(nodeList.size());
            excludeNodes = new ArrayList<>(nodeList.size());
            nodeAll:
            for (T t : nodeList) {
                for (T filterNode : filterNodes) {
                    // 判断是否属于过滤掉节点的子孙节点
                    if (filterNode.getRootId().longValue() == t.getRootId() &&
                            t.getLft() > filterNode.getLft() && t.getRgt() < filterNode.getRgt()) {
                        excludeNodes.add(t);
                        continue nodeAll;
                    }
                }

                if (nodePredicate == null || nodePredicate.test(t)) {
                    candidateNodes.add(t);
                    continue;
                }
                excludeNodes.add(t);
            }

            nodeList = candidateNodes;
            if (excludeNodes.isEmpty()) {
                // 已没有需要过滤的节点
                break;
            }
            filterNodes.addAll(excludeNodes);
        }
        return nodeList;
    }

    private void moveNode(@NotNull T node, TreeNodeBO parentNode, TreeNodeBO leftNode) {
        if (leftNode != null) {
            moveNodeByLeftNode(node, leftNode);
            return;
        }
        if (parentNode != null) {
            moveNodeByParentNode(node, parentNode);
            return;
        }

        // 移至根节点
        moveNodeByRoot(node);
    }

    private void moveNodeByLeftNode(@NotNull T node, @NotNull TreeNodeBO leftNode) {
        if (node.getId().longValue() == leftNode.getId()) {
            log.warn("树节点移动失败，要移动的节点与左节点相同");
            return;
        }
        if (node.getPId().longValue() == leftNode.getPid() && node.getLft() - leftNode.getRgt() == 1) {
            // 已在指定位置，无需移动
            return;
        }

        // 先删除原节点，再新增新节点
        removeTreeNode(node);
        addNode(node, leftNode.getPid(), leftNode.getId());
    }

    private void moveNodeByParentNode(@NotNull T node, @NotNull TreeNodeBO parentNode) {
        if (node.getPId().longValue() == parentNode.getId()) {
            log.warn("已在指定子节点下，无需移动");
            return;
        }

        // 先删除原节点，再新增新节点
        removeTreeNode(node);
        addNode(node, parentNode.getId(), null);
    }

    private void moveNodeByRoot(@NotNull T node) {
        // 先删除原节点，再新增新节点
        removeTreeNode(node);
        addRootNode(node);
    }

    private void addRootNode(@NotNull T node) {
        node.setLft(1);
        node.setRgt(2);
        node.setPId(BaseTreeModel.DEFAULT_PARENT);
        node.setRootId(node.getId());
        node.setDepth(1);
        jpaQueryFactory.update(qModel)
                .set(getLftPath(), node.getLft())
                .set(getRgtPath(), node.getRgt())
                .set(getPidPath(), BaseTreeModel.DEFAULT_PARENT)
                .set(getRootPath(), node.getRootId())
                .set(getDepthPath(), node.getDepth())
                .where(getIdPath().eq(node.getId()))
                .execute();
    }

    private void addNode(@NotNull T node, @NotNull Long parentId, Long leftNodeId) {
        if (parentId == BaseTreeModel.DEFAULT_PARENT) {
            // 增加根节点
            addRootNode(node);
            return;
        }

        // 增加的是子节点
        TreeNodeBO parentNode = null, leftNode = null;
        Set<Long> tempIds = leftNodeId == null ? Set.of(parentId) : Set.of(parentId, leftNodeId);
        var nodes = getNodeInfo(tempIds);
        parentNode = nodes.get(parentId);
        Assert.notNull(parentNode, "父节点不存在");
        Assert.notNull(parentNode.getLft(), "父节点未初始化");
        if (parentNode.getRgt() - parentNode.getLft() > 1 && leftNodeId != null) {
            // 获取左节点
            leftNode = nodes.get(leftNodeId);
            Assert.notNull(leftNode, "左节点不存在");
        }
        this.addChildNode(node, parentNode, leftNode);
    }

    private void addChildNode(@NotNull T node, @NotNull TreeNodeBO parentNode, TreeNodeBO leftNode) {
        if (parentNode.getRgt() - parentNode.getLft() > 1 && leftNode == null) {
            // 获取默认的左节点
            leftNode = getLastChildNode(parentNode.getId());
            Assert.notNull(leftNode, "获取左节点失败");
        }

        var pathRoot = getRootPath();
        var pathLft = getLftPath();
        var pathRgt = getRgtPath();

        // 生成左右节点值
        // 如果是第一个子节点，则左节点值取父节点的rgt，否则取左节点的rgt + 1
        int lft, rgt;
        if (leftNode == null) {
            // 作为第一个子节点
            lft = parentNode.getRgt();
            // 更新同树的lft
            jpaQueryFactory.update(qModel)
                    .set(pathLft, pathLft.add(2))
                    .where(pathRoot.eq(parentNode.getRootId()).and(pathLft.gt(lft)))
                    .execute();
            // 更新同树的rgt
            jpaQueryFactory.update(qModel)
                    .set(pathRgt, pathRgt.add(2))
                    .where(pathRoot.eq(parentNode.getRootId()).and(pathRgt.goe(lft)))
                    .execute();
        } else {
            // 不是第一个子节点
            lft = leftNode.getRgt() + 1;
            // 更新同树的lft
            jpaQueryFactory.update(qModel)
                    .set(pathLft, pathLft.add(2))
                    .where(pathRoot.eq(parentNode.getRootId()).and(pathLft.goe(lft)))
                    .execute();
            // 更新同树的rgt
            jpaQueryFactory.update(qModel)
                    .set(pathRgt, pathRgt.add(2))
                    .where(pathRoot.eq(parentNode.getRootId()).and(pathRgt.goe(lft)))
                    .execute();
        }
        rgt = lft + 1;

        // 更新当前节点
        node.setLft(lft);
        node.setRgt(rgt);
        node.setRootId(parentNode.getRootId());
        node.setPId(parentNode.getId());
        node.setDepth(parentNode.getDepth() + 1);
        jpaQueryFactory.update(qModel)
                .set(pathLft, node.getLft())
                .set(pathRgt, node.getRgt())
                .set(pathRoot, node.getRootId())
                .set(getPidPath(), parentNode.getId())
                .set(getDepthPath(), node.getDepth())
                .where(getIdPath().eq(node.getId()))
                .execute();
    }

    private void addNodeWithChildren(List<T> nodes, Long pid, Map<Long, List<T>> nodesMap) {
        Long tempLeftNodeId = null;
        int sortNo = 1;
        for (T node : nodes) {
            if (node.getSortNo() == null) {
                node.setSortNo(1);
            }
            // 添加节点
            this.addNode(node, pid, tempLeftNodeId);
            // 添加子节点
            var children = nodesMap.getOrDefault(node.getId(), Collections.emptyList());
            if (!children.isEmpty()) {
                this.addNodeWithChildren(children, node.getId(), nodesMap);
            }

            tempLeftNodeId = node.getId();
            sortNo++;
        }
    }

    private Long clearNodeInfo(Long nodeId) {
        if (nodeId == null) {
            // 没有指定节点，则清理所有树节点
            jpaQueryFactory.update(qModel)
                    .setNull(getPidPath())
                    .setNull(getLftPath())
                    .setNull(getRgtPath())
                    .execute();
            return null;
        }

        // 获取根节点id
        var rootId = jpaQueryFactory.select(getRootPath())
                .from(qModel)
                .where(getIdPath().eq(nodeId))
                .limit(1)
                .fetchOne();
        Assert.notNull(rootId, "未知节点[" + nodeId + "]所在树");
        jpaQueryFactory.update(qModel)
                .setNull(getPidPath())
                .setNull(getLftPath())
                .setNull(getRgtPath())
                .where(getRootPath().eq(rootId))
                .execute();

        return rootId;
    }

    private TreeNodeBO getNodeInfo(Long id) {
        if (id == BaseTreeModel.DEFAULT_PARENT) {
            return null;
        }
        return jpaQueryFactory.select(qBeanOfTreeNodeBO())
                .from(qModel)
                .where(getIdPath().eq(id))
                .limit(1)
                .fetchOne();
    }

    private Map<Long, TreeNodeBO> getNodeInfo(Set<Long> ids) {
        return jpaQueryFactory.select(qBeanOfTreeNodeBO())
                .from(qModel)
                .where(getIdPath().in(ids))
                .fetch()
                .stream()
                .collect(Collectors.toMap(TreeNodeBO::getId, t -> t, (t1, t2) -> t1))
                ;
    }

    private TreeNodeBO getLastChildNode(Long pId) {
        if (pId == BaseTreeModel.DEFAULT_PARENT) {
            return null;
        }
        return jpaQueryFactory.select(qBeanOfTreeNodeBO())
                .from(qModel)
                .where(getPidPath().eq(pId))
                .orderBy(getRgtPath().desc())
                .limit(1)
                .fetchOne();
    }

    private TreeNodeBO getFirstChildNode(Long pId) {
        if (pId == BaseTreeModel.DEFAULT_PARENT) {
            return null;
        }
        return jpaQueryFactory.select(qBeanOfTreeNodeBO())
                .from(qModel)
                .where(getPidPath().eq(pId))
                .orderBy(getLftPath().asc())
                .limit(1)
                .fetchOne();
    }

    private QBean<TreeNodeBO> qBeanOfTreeNodeBO() {
        return Projections.bean(TreeNodeBO.class, getIdPath(), getPidPath().as("pid"), getRootPath(), getLftPath(), getRgtPath(), getDepthPath());
    }

    private NumberPath<Integer> getLftPath() {
        return (NumberPath<Integer>) pathCache.computeIfAbsent(PROPERTY_LFT, p -> Expressions.numberPath(Integer.class, qModel, p));
    }

    private NumberPath<Integer> getRgtPath() {
        return (NumberPath<Integer>) pathCache.computeIfAbsent(PROPERTY_RGT, p -> Expressions.numberPath(Integer.class, qModel, p));
    }

    private NumberPath<Long> getRootPath() {
        return (NumberPath<Long>) pathCache.computeIfAbsent(PROPERTY_ROOT, p -> Expressions.numberPath(Long.class, qModel, p));
    }

    private NumberPath<Integer> getDepthPath() {
        return (NumberPath<Integer>) pathCache.computeIfAbsent(PROPERTY_DEPTH, p -> Expressions.numberPath(Integer.class, qModel, p));
    }

    private NumberPath<Long> getPidPath() {
        return (NumberPath<Long>) pathCache.computeIfAbsent(PROPERTY_PID, p -> Expressions.numberPath(Long.class, qModel, p));
    }

    private NumberPath<Integer> getSortNoPath() {
        return (NumberPath<Integer>) pathCache.computeIfAbsent(PROPERTY_SORT_NO, p -> Expressions.numberPath(Integer.class, qModel, p));
    }

    private Map<String, Path<?>> pathCache = new ConcurrentHashMap<>(16);
    private static final String PROPERTY_ID = "id";
    private static final String PROPERTY_LFT = "lft";
    private static final String PROPERTY_RGT = "rgt";
    private static final String PROPERTY_ROOT = "rootId";
    private static final String PROPERTY_DEPTH = "depth";
    private static final String PROPERTY_PID = "pId";
    private static final String PROPERTY_SORT_NO = "sortNo";
}
