package com.elitescloud.boot.jpa.config.auditing;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.context.TenantOrgContext;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.auditing.AuditingHandler;
import org.springframework.data.mapping.PersistentEntity;
import org.springframework.data.mapping.PersistentProperty;
import org.springframework.data.mapping.context.MappingContext;
import org.springframework.data.mapping.context.PersistentEntities;

import javax.annotation.Nonnull;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * 自定义审计处理器.
 *
 * @author Kaiser（wang shao）
 * @date 2021-03-04
 */
@Slf4j
public class CustomAuditHandler extends AuditingHandler {

    private TenantClientProvider tenantClientProvider;
    private UserDetailProvider userProvider;
    private List<AuditCustomizer> customizers;

    public CustomAuditHandler(MappingContext<? extends PersistentEntity<?, ?>, ? extends PersistentProperty<?>> mappingContext) {
        super(mappingContext);
    }

    public CustomAuditHandler(PersistentEntities entities) {
        super(entities);
    }

    @Nonnull
    @Override
    public <T> T markCreated(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // 创建时间
            LocalDateTime now = entity.getCreateTime() == null ? LocalDateTime.now() : entity.getCreateTime();

            // 租户id
            Long tenantId = entity.getTenantId() == null ? this.obtainTenantId() : entity.getTenantId();
            // 租户组织ID
            Long tenantOrgId = entity.getTenantOrgId() == null ? this.obtainTenantOrgId() : entity.getTenantOrgId();
            // 组织
            Long orgId = ObjectUtil.defaultIfNull(entity.getBelongOrgId(), entity.getSecBuId());

            // 当前操作用户
            var user = currentUser();
            Long currentUserId = 0L;
            String currentUsername = null;
            if (user != null && user.getUser() != null) {
                currentUserId = user.getUserId();
                currentUsername = user.getUser().getPrettyName();
                orgId = orgId == null ? user.getOrgId() : orgId;
            }
            if (entity.getCreateUserId() != null && CharSequenceUtil.isNotBlank(entity.getCreator())) {
                currentUserId = entity.getCreateUserId();
                currentUsername = entity.getCreator();
            }

            // 设置审计字段
            entity.setCreateUserId(currentUserId);
            entity.setCreator(currentUsername);
            entity.setCreateTime(now);
            if (entity.getModifyUserId() == null || CharSequenceUtil.isBlank(entity.getUpdater())) {
                entity.setModifyUserId(currentUserId);
                entity.setUpdater(currentUsername);
            }
            entity.setModifyTime(ObjectUtil.defaultIfNull(entity.getModifyTime(), now));
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
            entity.setTenantId(tenantId);
            entity.setTenantOrgId(tenantOrgId);
            entity.setBelongOrgId(orgId);
        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onCreate(source));
        return source;
    }

    @Nonnull
    @Override
    public <T> T markModified(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // 更新用户
            var user = currentUser();
            if (user != null && user.getUser() != null) {
                entity.setModifyUserId(user.getUserId());
                entity.setUpdater(user.getUser().getPrettyName());
            }

            entity.setModifyTime(LocalDateTime.now());
            entity.setDeleteFlag(CommonConstant.COMMON_DELETE_NO);
        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onUpdate(source));
        return source;
    }

    private Long obtainTenantId() {
        var currentTenant = getTenantClientProvider().getSessionTenant();
        if (currentTenant != null) {
            return currentTenant.getId();
        }

        return TenantConstant.DEFAULT_TENANT_ID;
    }

    private Long obtainTenantOrgId() {
        var useTenantOrg = TenantOrgContext.isUseTenantOrg();
        if (!ObjectUtil.defaultIfNull(useTenantOrg, true)) {
            // 不使用租户组织
            return null;
        }

        return TenantOrgContext.getTenantOrg();
    }

    private GeneralUserDetails currentUser() {
        if (userProvider == null) {
            userProvider = SpringContextHolder.getBean(UserDetailProvider.class);
        }
        return userProvider.currentUser();
    }

    public TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }

    public List<AuditCustomizer> getCustomizers() {
        if (customizers == null) {
            customizers = SpringContextHolder.getObjectProvider(AuditCustomizer.class).stream().collect(Collectors.toList());
        }
        return customizers;
    }
}
