package com.elitescloud.boot.jpa.support.id.provider.uidgenerator.worker;

import cn.hutool.core.util.IdUtil;
import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.jpa.support.id.config.GenType;
import com.elitescloud.boot.jpa.support.id.config.IdProperties;
import lombok.extern.log4j.Log4j2;
import org.springframework.dao.DataAccessException;
import org.springframework.jdbc.core.JdbcTemplate;

import java.time.LocalDateTime;

/**
 * 基于数据库的workerId分配器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/12/9
 */
@Log4j2
public class DatabaseWorkerIdAssigner extends AbstractWorkerIdAssigner {
    private static final String SQL_INSERT = "insert into sys_id_generator(id, gen_type, data_center_name, data_center_id, worker_id, instance_ip, create_time, modify_time) value (" +
            "?, ?, ?, ?, ?, ?, ?, ?)";
    private static final String SQL_QUERY_WORKER_ID = "select worker_id from sys_id_generator ";
    private static final String SQL_REFRESH = "update sys_id_generator set modify_time = ? where data_center_id = ? and worker_id = ?";
    private static final String SQL_DELETE_EXPIRE = "delete from sys_id_generator where modify_time < ?";
    private static final String SQL_DESTROY = "delete from sys_id_generator where data_center_id = ? and worker_id = ?";

    private static final int RETRY_FACTORY = 100;
    private static final int DATA_CENTER_ID = -1;
    private final IdProperties idProperties;
    private final JdbcTemplate jdbcTemplate;
    private final String dataCenterName;

    private long workerId;

    public DatabaseWorkerIdAssigner(IdProperties idProperties, JdbcTemplate jdbcTemplate, String dataCenterName) {
        this.idProperties = idProperties;
        this.jdbcTemplate = jdbcTemplate;
        this.dataCenterName = dataCenterName;
    }

    @Override
    public void refreshAlive() {
        LocalDateTime now = LocalDateTime.now();
        try {
            jdbcTemplate.update(SQL_REFRESH, now, DATA_CENTER_ID, workerId);

            // 删除过期的
            jdbcTemplate.update(SQL_DELETE_EXPIRE, now.minusMinutes(idProperties.getAlive().toMinutes()));
        } catch (DataAccessException e) {
            log.error("刷新UID workerIdAssigner异常：", e);
        }
    }

    @Override
    public void destroy() {
        try {
            jdbcTemplate.update(SQL_DESTROY, DATA_CENTER_ID, workerId);
        } catch (Exception ignored) {
        }
    }

    @Override
    public long assignWorkerId() {
        var id = generateWorkerId();
        this.workerId = id;
        return id;
    }

    private long generateWorkerId() {
        // 最大取值
        long max = super.maxWorkerId(idProperties);
        // 查询出所有已存在的值
        var existsWorkerIds = jdbcTemplate.queryForList(SQL_QUERY_WORKER_ID, Long.class);

        int retryTimes = 0;
        String ip = super.getIp();
        while (true) {
            long temp = RandomUtil.randomLong(max);
            if (existsWorkerIds.contains(temp)) {
                // 已存在
                continue;
            }
            var now = LocalDateTime.now();
            try {
                jdbcTemplate.update(SQL_INSERT, IdUtil.fastSimpleUUID(), GenType.UID_GENERATOR.name(), dataCenterName, DATA_CENTER_ID, temp,
                        ip, now, now);

                return temp;
            } catch (DataAccessException e) {
                retryTimes++;
            }

            if (retryTimes >= RETRY_FACTORY) {
                log.warn("尝试使用WorkerId第{}次", retryTimes);
            }
        }
    }
}
