package com.elitescloud.boot.jpa.config.auditing;

import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.context.TenantOrgContext;
import com.elitescloud.boot.jpa.CloudtDataProperties;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import lombok.extern.slf4j.Slf4j;

import javax.annotation.Nonnull;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * @author : chen.niu
 * @description : 自定义审计处理器.工具类
 * @date : 2024-04-02 10:36
 */
@Slf4j
public class CustomAuditUtil {
    private static TenantClientProvider tenantClientProvider;
    private static UserDetailProvider userProvider;
    private static CloudtDataProperties dataProperties;
    private static List<AuditCustomizer> customizers;

    public static <T> T markCreated(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // 创建时间
            LocalDateTime now = entity.getCreateTime() == null ? LocalDateTime.now() : entity.getCreateTime();

            // 租户id
            Long tenantId = entity.getTenantId() == null ? obtainTenantId() : entity.getTenantId();
            // 租户组织ID
            Long tenantOrgId = entity.getTenantOrgId() == null ? obtainTenantOrgId() : entity.getTenantOrgId();
            // 组织
            Long orgId = ObjectUtil.defaultIfNull(entity.getBelongOrgId(), entity.getSecBuId());

            // 当前操作用户
            var user = currentUser();
            Long currentUserId = 0L;
            String currentUsername = null;
            if (user != null && user.getUser() != null) {
                currentUserId = user.getUserId();
                currentUsername = obtainCurrentUserName(user);
                orgId = orgId == null ? user.getOrgId() : orgId;
            }
            if (entity.getCreateUserId() != null && CharSequenceUtil.isNotBlank(entity.getCreator())) {
                currentUserId = entity.getCreateUserId();
                currentUsername = entity.getCreator();
            }

            // 设置审计字段
            entity.setCreateUserId(currentUserId);
            entity.setCreator(currentUsername);
            entity.setCreateTime(now);
            if (entity.getModifyUserId() == null || CharSequenceUtil.isBlank(entity.getUpdater())) {
                entity.setModifyUserId(currentUserId);
                entity.setUpdater(currentUsername);
            }
            entity.setModifyTime(ObjectUtil.defaultIfNull(entity.getModifyTime(), now));
            entity.setDeleteFlag(ObjectUtil.defaultIfNull(entity.getDeleteFlag(), CommonConstant.COMMON_DELETE_NO));
            entity.setTenantId(tenantId);
            entity.setTenantOrgId(tenantOrgId);
            entity.setBelongOrgId(orgId);
        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onCreate(source));
        return source;
    }

    @Nonnull
    public static <T> T markModified(@Nonnull T source) {
        if (source instanceof BaseModel) {
            BaseModel entity = (BaseModel) source;

            // 更新用户
            var user = currentUser();
            if (user != null && user.getUser() != null) {
                entity.setModifyUserId(user.getUserId());
                entity.setUpdater(obtainCurrentUserName(user));
            }

            entity.setModifyTime(LocalDateTime.now());
            if (entity.getDeleteFlag() == null) {
                entity.setDeleteFlag(CommonConstant.COMMON_DELETE_NO);
            }
        }

        // 自定义处理
        getCustomizers().forEach(t -> t.onUpdate(source));
        return source;
    }


    protected static String obtainCurrentUserName(GeneralUserDetails currentUser) {
        if (currentUser == null) {
            currentUser = currentUser();
        }
        if (currentUser == null) {
            return null;
        }

        var auditorType = ObjectUtil.defaultIfNull(getDataProperties().getAuditorType(), CloudtDataProperties.AuditorType.FULL_NAME);
        switch (auditorType) {
            case FULL_NAME:
                return currentUser.getUser().getPrettyName();
            case USER_NAME:
                return currentUser.getUsername();
            case CUSTOM:
                log.warn("暂不支持自定义");
                return currentUser.getUser().getPrettyName();
            default:
                return null;
        }
    }

    protected static Long obtainTenantId() {
        var currentTenant = getTenantClientProvider().getSessionTenant();
        if (currentTenant != null) {
            return currentTenant.getId();
        }

        return TenantConstant.DEFAULT_TENANT_ID;
    }

    protected  static Long obtainTenantOrgId() {
        var useTenantOrg = TenantOrgContext.isUseTenantOrg();
        if (!ObjectUtil.defaultIfNull(useTenantOrg, true)) {
            // 不使用租户组织
            return null;
        }

        return TenantOrgContext.getTenantOrg();
    }

    protected static GeneralUserDetails currentUser() {
        if (userProvider == null) {
            userProvider = SpringContextHolder.getBean(UserDetailProvider.class);
        }
        return userProvider.currentUser();
    }

    protected  static TenantClientProvider getTenantClientProvider() {
        if (tenantClientProvider == null) {
            tenantClientProvider = SpringContextHolder.getBean(TenantClientProvider.class);
        }
        return tenantClientProvider;
    }

    public static CloudtDataProperties getDataProperties() {
        if (dataProperties == null) {
            dataProperties = SpringContextHolder.getBean(CloudtDataProperties.class);
        }
        return dataProperties;
    }

    protected static List<AuditCustomizer> getCustomizers() {
        if (customizers == null) {
            customizers = SpringContextHolder.getObjectProvider(AuditCustomizer.class).stream().collect(Collectors.toList());
        }
        return customizers;
    }
}
