package com.elitescloud.boot.jpa.config;

import cn.hutool.core.util.ObjectUtil;
import com.blazebit.persistence.ConfigurationProperties;
import com.blazebit.persistence.Criteria;
import com.blazebit.persistence.CriteriaBuilderFactory;
import com.elitescloud.boot.jpa.CloudtDataProperties;
import com.elitescloud.boot.jpa.CloudtJpaProperties;
import com.elitescloud.boot.jpa.support.dialect.MySQL5Dialect;
import com.elitescloud.boot.model.entity.BaseModel;
import com.elitescloud.boot.provider.UserDetailProvider;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.querydsl.core.types.EntityPath;
import com.querydsl.core.types.dsl.Expressions;
import com.querydsl.jpa.impl.JPAQueryFactory;
import com.querydsl.jpa.impl.JPAUpdateClause;
import lombok.extern.slf4j.Slf4j;
import org.hibernate.cfg.AvailableSettings;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.beans.factory.config.ConfigurableBeanFactory;
import org.springframework.boot.autoconfigure.orm.jpa.HibernatePropertiesCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Scope;

import javax.persistence.EntityManager;
import javax.persistence.EntityManagerFactory;
import java.time.LocalDateTime;

/**
 * @author Michael Li
 * @date 2020-09-19
 */
public class HibernateConfig {

    private final CloudtJpaProperties jpaProperties;

    public HibernateConfig(CloudtJpaProperties jpaProperties) {
        this.jpaProperties = jpaProperties;
    }

    @Bean
    public HibernatePropertiesCustomizer hibernatePropertiesCustomizer() {
        return hibernateProperties -> {
            // sql前的注释 log4jdbc会因为SQL前的注释中的占位符导致打印SQL参数出现错位
            hibernateProperties.putIfAbsent(AvailableSettings.USE_SQL_COMMENTS, false);
            // hibernate更新schema时唯一索引的策略，尝试创建，并忽略异常
            hibernateProperties.putIfAbsent(AvailableSettings.UNIQUE_CONSTRAINT_SCHEMA_UPDATE_STRATEGY, "RECREATE_QUIETLY");
            // 批量更新时的数量
            hibernateProperties.putIfAbsent(AvailableSettings.STATEMENT_BATCH_SIZE, 50);
            // 批量时顺序插入
            hibernateProperties.putIfAbsent(AvailableSettings.ORDER_INSERTS, "true");
            // 批量时顺序更新
            hibernateProperties.putIfAbsent(AvailableSettings.ORDER_UPDATES, "true");
            hibernateProperties.putIfAbsent(AvailableSettings.BATCH_VERSIONED_DATA, "true");
//            hibernateProperties.put("hibernate.integrator_provider",
//                    (IntegratorProvider) () -> Collections.singletonList(CommentIntegrator.INSTANCE));
//            hibernateProperties.put(AvailableSettings.STATEMENT_INSPECTOR, HibernateLogInterceptor.class.getName());
            hibernateProperties.putIfAbsent(AvailableSettings.DIALECT,
                    MySQL5Dialect.class.getName());
        };
    }

    @Bean
    public JPAQueryFactory jpaQueryFactory(EntityManager em,
                                           ObjectProvider<UserDetailProvider> userProvider,
                                           CloudtDataProperties dataProperties) {
        return new CloudtJPAQueryFactory(em, userProvider.getIfAvailable(), dataProperties);
    }

    @Bean
    @Scope(ConfigurableBeanFactory.SCOPE_SINGLETON)
    public CriteriaBuilderFactory criteriaBuilderFactory(EntityManagerFactory entityManagerFactory) {
        var criteria = Criteria.getDefault();
        criteria.setProperty(ConfigurationProperties.COMPATIBLE_MODE, jpaProperties.getBlaze().getCompatibleMode().toString());
        criteria.setProperty(ConfigurationProperties.RETURNING_CLAUSE_CASE_SENSITIVE, jpaProperties.getBlaze().getReturningClauseCaseSensitive().toString());
        criteria.setProperty(ConfigurationProperties.SIZE_TO_COUNT_TRANSFORMATION, jpaProperties.getBlaze().getSizeToCountTransformation().toString());
        criteria.setProperty(ConfigurationProperties.IMPLICIT_GROUP_BY_FROM_SELECT, jpaProperties.getBlaze().getImplicitGroupByFromSelect().toString());
        criteria.setProperty(ConfigurationProperties.IMPLICIT_GROUP_BY_FROM_HAVING, jpaProperties.getBlaze().getImplicitGroupByFromHaving().toString());
        criteria.setProperty(ConfigurationProperties.IMPLICIT_GROUP_BY_FROM_ORDER_BY, jpaProperties.getBlaze().getImplicitGroupByFromOrderBy().toString());
        criteria.setProperty(ConfigurationProperties.EXPRESSION_OPTIMIZATION, jpaProperties.getBlaze().getExpressionOptimization().toString());
        criteria.setProperty(ConfigurationProperties.EXPRESSION_CACHE_CLASS, jpaProperties.getBlaze().getCacheClass());
        criteria.setProperty(ConfigurationProperties.VALUES_CLAUSE_FILTER_NULLS, jpaProperties.getBlaze().getFilterNulls().toString());
        criteria.setProperty(ConfigurationProperties.PARAMETER_AS_LITERAL_RENDERING, jpaProperties.getBlaze().getParameterLiteralRendering().toString());
        criteria.setProperty(ConfigurationProperties.OPTIMIZED_KEYSET_PREDICATE_RENDERING, jpaProperties.getBlaze().getOptimizedKeysetPredicateRendering().toString());
        criteria.setProperty(ConfigurationProperties.INLINE_ID_QUERY, jpaProperties.getBlaze().getInlineIdQuery());
        criteria.setProperty(ConfigurationProperties.INLINE_COUNT_QUERY, jpaProperties.getBlaze().getInlineCountQuery());
        criteria.setProperty(ConfigurationProperties.INLINE_CTES, jpaProperties.getBlaze().getInlineCtes());
        criteria.setProperty(ConfigurationProperties.QUERY_PLAN_CACHE_ENABLED, jpaProperties.getBlaze().getQueryPlanCacheEnabled().toString());
        criteria.setProperty(ConfigurationProperties.CRITERIA_NEGATION_WRAPPER, jpaProperties.getBlaze().getCriteriaNegationWrapper().toString());
        criteria.setProperty(ConfigurationProperties.CRITERIA_VALUE_AS_PARAMETER, jpaProperties.getBlaze().getCriteriaValueAsParameter().toString());

        return criteria.createCriteriaBuilderFactory(entityManagerFactory);
    }

    @Slf4j
    static class CloudtJPAQueryFactory extends JPAQueryFactory {

        private final UserDetailProvider userProvider;
        private final CloudtDataProperties dataProperties;

        public CloudtJPAQueryFactory(EntityManager entityManager, UserDetailProvider userProvider, CloudtDataProperties dataProperties) {
            super(entityManager);
            this.userProvider = userProvider;
            this.dataProperties = dataProperties;
        }

        @Override
        public JPAUpdateClause update(EntityPath<?> path) {
            var clause = super.update(path);

            // 增加审计
            this.auditForUpdate(path, clause);
            return clause;
        }

        private void auditForUpdate(EntityPath<?> path, JPAUpdateClause clause) {
            if (BaseModel.class.isAssignableFrom(path.getType())) {
                var now = LocalDateTime.now();
                clause.set(Expressions.dateTimePath(LocalDateTime.class, path, "modifyTime"), now);

                // 当前用户
                var user = userProvider == null ? null : userProvider.currentUser();
                if (user != null) {
                    clause.set(Expressions.numberPath(Long.class, path, "modifyUserId"), user.getUserId());
                    clause.set(Expressions.stringPath(path, "updater"), this.obtainCurrentUserName(user));
                }
            }
        }

        private String obtainCurrentUserName(GeneralUserDetails currentUser) {
            if (currentUser == null) {
                return null;
            }

            var auditorType = ObjectUtil.defaultIfNull(dataProperties.getAuditorType(), CloudtDataProperties.AuditorType.FULL_NAME);
            switch (auditorType) {
                case FULL_NAME:
                    return currentUser.getUser().getPrettyName();
                case USER_NAME:
                    return currentUser.getUsername();
                case CUSTOM:
                    log.warn("暂不支持自定义");
                    return currentUser.getUser().getPrettyName();
                default:
                    return null;
            }
        }
    }
}
