package com.elitescloud.boot.tool;

import com.nimbusds.jose.JOSEException;
import com.nimbusds.jose.jwk.JWKSet;
import com.nimbusds.jose.jwk.RSAKey;
import com.nimbusds.jose.jwk.source.JWKSource;
import com.nimbusds.jose.proc.SecurityContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.Resource;
import org.springframework.lang.NonNull;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.security.oauth2.jwt.JwtEncoder;
import org.springframework.security.oauth2.jwt.NimbusJwtDecoder;
import org.springframework.security.oauth2.jwt.NimbusJwtEncoder;
import org.springframework.util.Assert;

import java.io.IOException;
import java.io.InputStream;
import java.security.KeyStore;

/**
 * 云梯jwt工具类.
 *
 * @author Kaiser（wang shao）
 * @date 2019/6/26
 */
public class JwtUtil {
    private static final Logger logger = LoggerFactory.getLogger(JwtUtil.class);

    /**
     * 构建jwt解密器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtDecoder buildJwtDecoder(@NonNull RSAKey rsaKey) {
        Assert.notNull(rsaKey, "rsaKey为空");

        try {
            return NimbusJwtDecoder.withPublicKey(rsaKey.toRSAPublicKey()).build();
        } catch (JOSEException e) {
            logger.error("jwtDecoder构建失败：", e);
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * 构建jwt加密器
     *
     * @param rsaKey rsaKey
     * @return decoder
     */
    public static JwtEncoder buildJwtEncoder(@NonNull RSAKey rsaKey) {
        Assert.notNull(rsaKey, "rsaKey为空");

        JWKSource<SecurityContext> jwkSource = generateJwkSource(rsaKey);
        return new NimbusJwtEncoder(jwkSource);
    }

    /**
     * 加载RSAKey
     *
     * @param keyStore keyStore
     * @param alias    key别名
     * @param secret   key Secret
     * @return RSAKey
     */
    public static RSAKey loadRSAKey(@NonNull KeyStore keyStore, String alias, @NonNull String secret) {
        Assert.notNull(keyStore, "keyStore为空");
        Assert.hasText(alias, "alias为空");
        Assert.hasText(secret, "secret为空");

        // 加载RSAKey
        try {
            return RSAKey.load(keyStore, alias, secret.toCharArray());
        } catch (Exception e) {
            logger.error("rsaKey加载失败：", e);
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreStream keyStore流
     * @param type           keyStore类型
     * @param password       keystore密码
     * @param alias          key别名
     * @param secret         key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull InputStream keystoreStream, String type, String password, String alias, String secret) {
        Assert.hasText(password, "password为空");
        Assert.hasText(alias, "alias为空");
        Assert.hasText(secret, "secret为空");

        KeyStore keyStore = null;
        try {
            keyStore = KeyStore.getInstance(type);
            keyStore.load(keystoreStream, password.toCharArray());
        } catch (Exception e) {
            logger.error("keystore加载失败：", e);
            throw new IllegalArgumentException(e);
        }

        return keyStore;
    }

    /**
     * 加载KeyStore工具
     *
     * @param keystoreResource keyStore资源
     * @param type             keyStore类型
     * @param password         keystore密码
     * @param alias            key别名
     * @param secret           key secret
     * @return KeyStore
     */
    public static KeyStore loadKeystore(@NonNull Resource keystoreResource, String type, @NonNull String password, String alias, String secret) {
        Assert.isTrue(keystoreResource.exists(), "keystore不存在");

        InputStream keystoreStream = null;
        try {
            keystoreStream = keystoreResource.getInputStream();
        } catch (IOException e) {
            logger.error("加载keystore失败：", e);
            throw new IllegalArgumentException("加载keystore失败", e);
        }

        return loadKeystore(keystoreStream, type, password, alias, secret);
    }

    /**
     * 生成JWKSource
     *
     * @param rsaKey rsaKey
     * @return jwkSource
     */
    public static JWKSource<SecurityContext> generateJwkSource(@NonNull RSAKey rsaKey) {
        JWKSet jwkSet = new JWKSet(rsaKey);
        return (jwkSelector, securityContext) -> jwkSelector.select(jwkSet);
    }
}
