package com.elitescloud.boot.tool;

import com.nimbusds.jose.jwk.RSAKey;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.io.ClassPathResource;
import org.springframework.security.oauth2.jwt.Jwt;
import org.springframework.security.oauth2.jwt.JwtDecoder;
import org.springframework.util.Assert;
import org.springframework.util.StringUtils;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;

/**
 * Token解析器.
 *
 * @author Kaiser（wang shao）
 * @date 2024/6/26
 */
public class TokenParser {
    private static final Logger logger = LoggerFactory.getLogger(TokenParser.class);
    private final JwtDecoder jwtDecoder;

    private TokenParser(RSAKey rsaKey) {
        this.jwtDecoder = JwtUtil.buildJwtDecoder(rsaKey);
    }

    /**
     * 获取默认token解析器
     *
     * @return token解析器
     */
    public static TokenParser instance() {
        RSAKey rsa = null;
        try {
            var classPathResource = new ClassPathResource("yst-config.jks");
            var keystore = JwtUtil.loadKeystore(classPathResource.getInputStream(), "jks", "elitesland516", "yst-config", "elitesland516");
            rsa = JwtUtil.loadRSAKey(keystore, "yst-config", "elitesland516");
        } catch (Exception e) {
            logger.error("秘钥库读取异常：", e);
            throw new IllegalArgumentException("秘钥库读取异常", e);
        }
        return new TokenParser(rsa);
    }

    /**
     * 自定义rsaKey的token解析器
     *
     * @param rsaKey 自定义rsaKey
     * @return token解析器
     */
    public static TokenParser instance(@NotNull RSAKey rsaKey) {
        Assert.notNull(rsaKey, "RSAKey为空");
        return new TokenParser(rsaKey);
    }

    /**
     * 解析token
     *
     * @param token token
     * @return payload
     */
    public CloudtJwt parse(@NotBlank String token) {
        Assert.hasText(token, "token为空");
        var jwt = jwtDecoder.decode(token);
        return new CloudtJwt(jwt);
    }

    public static class CloudtJwt {
        private final Jwt jwt;

        private CloudtJwt(Jwt jwt) {
            this.jwt = jwt;
        }

        /**
         * 获取原始jwt信息
         *
         * @return jwt
         */
        public Jwt getOriginalJwt() {
            return jwt;
        }

        /**
         * 获取登录号
         *
         * @return 登录号
         */
        public String getUsername() {
            return jwt.getSubject();
        }

        /**
         * 获取用户ID
         *
         * @return 用户ID
         */
        public Long getUserId() {
            return this.getClaimAsLong("yst_ui");
        }

        /**
         * 获取用户ID
         *
         * @return 用户ID
         */
        public Long getTenantId() {
            return this.getClaimAsLong("yst_ti");
        }

        private Long getClaimAsLong(String claim) {
            var claimValue = jwt.getClaimAsString(claim);
            if (!StringUtils.hasText(claimValue)) {
                return null;
            }
            return Long.parseLong(claimValue);
        }
    }
}
