package com.elitescloud.boot.log;

import com.elitescloud.boot.log.config.AccessLogConfig;
import com.elitescloud.boot.log.config.LogStorageConfig;
import com.elitescloud.boot.log.config.LoginLogConfig;
import com.elitescloud.boot.log.common.LogStorable;
import com.elitescloud.boot.log.queue.LogEvent;
import com.elitescloud.boot.log.queue.LogEventFactory;
import com.elitescloud.boot.log.queue.LogEventHandler;
import com.lmax.disruptor.BlockingWaitStrategy;
import com.lmax.disruptor.ExceptionHandler;
import com.lmax.disruptor.RingBuffer;
import com.lmax.disruptor.dsl.Disruptor;
import com.lmax.disruptor.dsl.ProducerType;
import lombok.extern.log4j.Log4j2;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.scheduling.concurrent.ThreadPoolExecutorFactoryBean;

/**
 * 日志相关自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 3/21/2023
 */
@ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(LogProperties.class)
@Import({LogStorageConfig.class, LoginLogConfig.class, AccessLogConfig.class})
@Log4j2
class CloudtLogAutoConfiguration {

    private final LogProperties logProperties;

    public CloudtLogAutoConfiguration(LogProperties logProperties) {
        this.logProperties = logProperties;
    }

    /**
     * 日志处理队列
     *
     * @param storable
     * @return
     */
    @Bean("logEventRingBuffer")
    public RingBuffer<LogEvent> logEventRingBuffer(LogStorable storable) {
        // 必须为2的N次方
        int bufferSize = 1024 * 256;

        // 线程池
        ThreadPoolExecutorFactoryBean executorFactoryBean = new ThreadPoolExecutorFactoryBean();
        var threadPool = logProperties.getThreadPool();
        executorFactoryBean.setThreadNamePrefix(threadPool.getThreadNamePrefix());
        executorFactoryBean.setCorePoolSize(threadPool.getCoreSize());
        executorFactoryBean.setMaxPoolSize(threadPool.getMaxSize());

        Disruptor<LogEvent> disruptor = new Disruptor<>(new LogEventFactory(), bufferSize, executorFactoryBean,
                ProducerType.SINGLE, new BlockingWaitStrategy());
        disruptor.handleEventsWith(new LogEventHandler(storable));
        disruptor.setDefaultExceptionHandler(new ExceptionHandler<>() {
            @Override
            public void handleEventException(Throwable ex, long sequence, LogEvent event) {
                log.error("日志处理异常：", ex);
            }

            @Override
            public void handleOnStartException(Throwable ex) {
                log.error("日志启动处理异常：", ex);
            }

            @Override
            public void handleOnShutdownException(Throwable ex) {
                log.error("日志结束处理异常：", ex);
            }
        });

        disruptor.start();
        return disruptor.getRingBuffer();
    }
}
