package com.elitescloud.boot.log.config;

import cn.hutool.core.text.CharSequenceUtil;
import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.json.jackson.JacksonJsonpMapper;
import co.elastic.clients.transport.ElasticsearchTransport;
import co.elastic.clients.transport.rest_client.RestClientTransport;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.common.LogStorable;
import com.elitescloud.boot.log.provider.storage.DatabaseStorage;
import com.elitescloud.boot.log.provider.storage.ElasticsearchStorage;
import com.elitescloud.boot.log.provider.storage.EmptyStorage;
import com.elitescloud.boot.log.provider.storage.LogPrintStorage;
import lombok.extern.log4j.Log4j2;
import org.apache.http.HttpHost;
import org.elasticsearch.client.RestClient;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;

/**
 * 日志持久化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/22
 */
@Import({LogStorageConfig.ElasticsearchStorageConfig.class, LogStorageConfig.DatabaseStorageConfig.class})
@Log4j2
public class LogStorageConfig {

    private final LogProperties logProperties;

    public LogStorageConfig(LogProperties logProperties) {
        this.logProperties = logProperties;
    }

    @ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".repository", name = "storage-type", havingValue = "log")
    @Bean
    public LogStorable logPrintStorage() {
        log.info("日志持久化方式：{}", logProperties.getRepository().getStorageType());
        return new LogPrintStorage(logProperties);
    }

    /**
     * 日志持久化
     *
     * @return
     */
    @Bean
    @ConditionalOnMissingBean
    public LogStorable logStorable() {
        log.info("日志持久化方式：{}", "不存储");
        return new EmptyStorage();
    }

    @ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".repository", name = "storage-type", havingValue = "database")
    static class DatabaseStorageConfig {
        private final LogProperties logProperties;

        public DatabaseStorageConfig(LogProperties logProperties) {
            this.logProperties = logProperties;
        }

        @Bean
        public LogStorable databasePrintStorage() {
            log.info("日志持久化方式：{}", logProperties.getRepository().getStorageType());

            return new DatabaseStorage(logProperties);
        }
    }

    /**
     * Elasticsearch持久化配置
     */
    @ConditionalOnClass({ElasticsearchClient.class})
    @ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".repository", name = "storage-type", havingValue = "elasticsearch")
    static class ElasticsearchStorageConfig {

        @Bean
        public LogStorable logStorableElasticsearch(LogProperties logProperties,
                                                    Jackson2ObjectMapperBuilder objectMapperBuilder) {
            log.info("日志持久化方式：{}", logProperties.getRepository().getStorageType());
            var elasticsearch = logProperties.getRepository().getElasticsearch();

            // 获取es的httpHost
            HttpHost[] httpHosts = new HttpHost[0];
            if (!CollectionUtils.isEmpty(elasticsearch.getUris())) {
                httpHosts = elasticsearch.getUris()
                        .stream()
                        .filter(CharSequenceUtil::isNotBlank)
                        .map(HttpHost::create).toArray(HttpHost[]::new);
            }
            Assert.notEmpty(httpHosts, "日志存储的Elasticsearch地址未配置");

            // 创建客户端
            JacksonJsonpMapper jsonpMapper = new JacksonJsonpMapper(objectMapperBuilder.build());
            ElasticsearchTransport transport = new RestClientTransport(RestClient.builder(httpHosts).build(), jsonpMapper);
            ElasticsearchClient elasticsearchClient = new ElasticsearchClient(transport);

            return new ElasticsearchStorage(logProperties, elasticsearchClient);
        }
    }
}
