package com.elitescloud.boot.log.interceptor;

import com.elitescloud.boot.log.common.BusinessObject;
import com.elitescloud.boot.log.common.BusinessObjectOperation;
import com.elitescloud.boot.log.model.bo.OperationLogDTO;
import com.elitescloud.boot.log.service.OperationLogMqMessageService;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;
import org.springframework.web.bind.annotation.RestController;

import java.time.Duration;
import java.time.Instant;

/**
 * @author : chen.niu
 * @description :
 * @date : 2023-12-11 21:15
 */
@Aspect
@Slf4j
public class BusinessOperationAspect {
    //自动
    public static final String AUTO = "auto";
    private final OperationLogMqMessageService operationLogService;
    private final ObjectMapper objectMapper = new ObjectMapper();
    //业务对象的默认值 表示没有单独设置业务对象，使用业务对象注解内容
    public static final String BUSINESS_OBJECT_NAME_DEFAULT = "@BusinessObject";

    public BusinessOperationAspect(OperationLogMqMessageService operationLogService) {
        this.operationLogService = operationLogService;
    }


    @Around("@annotation(businessObjectOperation)")
    public Object logBusinessOperation(ProceedingJoinPoint joinPoint, BusinessObjectOperation businessObjectOperation) throws Throwable {
//        BusinessOperationAnnotation annotation = obtainAnnotation(joinPoint);
        // 检查是否在带有 @RestController 的类的方法上
        if (!joinPoint.getTarget().getClass().isAnnotationPresent(RestController.class)) {
            log.error("注解位置异常：{} BusinessOperationAnnotation 规范应用在  @RestController classes的方法中"
                    , joinPoint.getTarget().getClass().getName());
        }

        if (BUSINESS_OBJECT_NAME_DEFAULT.equals(businessObjectOperation.businessObjectType())) {
            if (!joinPoint.getTarget().getClass().isAnnotationPresent(BusinessObject.class)) {
                log.error("注解异常: 跳过，继续执行业务代码：{} BusinessOperationAnnotation 规范应用在  @BusinessObjectAnnotation classes的方法中"
                        , joinPoint.getTarget().getClass().getName());
                return joinPoint.proceed();
            }
        }
        BusinessObject businessObject = joinPoint.getTarget().getClass().getAnnotation(BusinessObject.class);

        if (businessObjectOperation.logEnabled()) {
            Instant start = Instant.now();

            String argsJson = null;
            if (businessObjectOperation.argsJsonEnabled()) {
                // 转换方法参数为JSON
                argsJson = convertToJson(joinPoint.getArgs());
            }
            try {
                // 执行目标方法
                Object result = joinPoint.proceed();
                long timeElapsed = Duration.between(start, Instant.now()).toMillis();
                // 记录操作日志
                logOperation(result,argsJson, timeElapsed,businessObject, businessObjectOperation);
                return result;
            } catch (Exception e) {
                long timeElapsed = Duration.between(start, Instant.now()).toMillis();
                 // 记录异常操作日志
                logOperationError(e,argsJson, timeElapsed,businessObject, businessObjectOperation);
                throw e;
            }

        } else {
            return joinPoint.proceed();
        }
    }





    @Around("@within(businessObjectAnnotation)")
    public Object checkBusinessObjectAnnotation(ProceedingJoinPoint joinPoint, BusinessObject businessObjectAnnotation) throws Throwable {
        // 检查是否在带有 @RestController 的类上
        if (!joinPoint.getTarget().getClass().isAnnotationPresent(RestController.class)) {
            log.error("注解位置异常：{} BusinessObjectAnnotation 规范应用在 @RestController classes"
                    , joinPoint.getTarget().getClass().getName());
        }
        return joinPoint.proceed();
    }

    /**
     * 根据业务对象操作注解，获取业务对象注解上的业务对象类型名称，如果是指定了业务对象，从指定的业务对此操作上直接获取。
     * @param businessObjectOperation
     * @param businessObject
     * @return
     */
    private static String getBusinessType(BusinessObjectOperation businessObjectOperation, BusinessObject businessObject) {
        String businessType;
        if (BUSINESS_OBJECT_NAME_DEFAULT.equals(businessObjectOperation.businessObjectType())) {
            businessType = businessObject.businessType();
        } else {
            //如果业务操作独立设置了业务对象
            businessType = businessObjectOperation.businessObjectType();
        }
        return businessType;
    }

    private void logOperationError(Exception proceedErr, String argsJson, long timeElapsed, BusinessObject businessObject, BusinessObjectOperation businessObjectOperation) {
        try {

            //如果默认
            String businessType = getBusinessType(businessObjectOperation, businessObject);

            OperationLogDTO operationLogDTO = operationLogService.quickNewOperationLogDTO(
                    businessType,
                    businessObjectOperation.businessParam(),
                    businessObjectOperation.operationType(),
                    businessObjectOperation.operationDescription());
            operationLogDTO.setDelayed(timeElapsed);
            operationLogDTO.setOperationAfterData(argsJson);
            operationLogDTO.setOperationBeforeData(proceedErr.toString());
            operationLogDTO.setSource(AUTO);
            operationLogDTO.setSuccess(false);
            operationLogDTO.setExceptionDesc("接口执行异常："+proceedErr.getMessage());
            ApiResult<String> apiResult = operationLogService.sendAsyncOperationLogMqMessage(operationLogDTO);
            if (!apiResult.isSuccess()) {
                log.error("操作日志：消息发送失败 - {}", apiResult.getErrorMsg());
            }
        } catch (Exception e) {
            log.error("操作日志：记录失败 - {}", e.getMessage(), e);
        }
    }

    private void logOperation(  Object resultData,String argsJson, long timeElapsed,
                                BusinessObject businessObject,BusinessObjectOperation businessObjectOperation) {
        try {

            String resultJson = null;
            if (businessObjectOperation.resultJsonEnabled()) {
                // 转换结果为JSON
                resultJson = convertToJson(resultData);
            }

            //如果默认
            String  businessType = getBusinessType(businessObjectOperation, businessObject);

            OperationLogDTO operationLogDTO = operationLogService.quickNewOperationLogDTO(
                    businessType,
                    businessObjectOperation.businessParam(),
                    businessObjectOperation.operationType(),
                    businessObjectOperation.operationDescription());
            operationLogDTO.setDelayed(timeElapsed);
            operationLogDTO.setOperationAfterData(argsJson);
            operationLogDTO.setOperationBeforeData(resultJson);
            operationLogDTO.setSource(AUTO);
            if(resultData instanceof ApiResult){
                ApiResult result = (ApiResult) resultData;
                operationLogDTO.setSuccess(result.isSuccess());
                StringBuffer messageBu = new StringBuffer();
                if(result.isSuccess()) {
                    messageBu.append(result.getCode()).append(";").append(result.getMsg());
                }else{
                    messageBu.append(result.getCode()).append(";").append(result.getErrorMsg())
                            .append(";").append(result.getErrorNo());
                }
                operationLogDTO.setExceptionDesc(messageBu.toString());
            }
            ApiResult<String> apiResult = operationLogService.sendAsyncOperationLogMqMessage(operationLogDTO);
            if (!apiResult.isSuccess()) {
                log.error("操作日志：消息发送失败 - {}", apiResult.getErrorMsg());
            }
        } catch (Exception e) {
            log.error("操作日志：记录失败 - {}", e.getMessage(), e);
        }
    }


    private String convertToJson(Object object) {
        try {
            return objectMapper.writeValueAsString(object);
        } catch (Exception e) {
            log.error("操作日志：JSON转换异常 - "+e.getMessage() , e);
            return "转换异常: " + e.getMessage();
        }
    }

    private BusinessObjectOperation obtainAnnotation(ProceedingJoinPoint point) {
        if (point instanceof MethodInvocationProceedingJoinPoint) {
            // 优先获取方法上的注解
            BusinessObjectOperation annotation = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(BusinessObjectOperation.class);
            return annotation;
        }
        return null;
    }
}

