package com.elitescloud.boot.log.provider.storage;

import co.elastic.clients.elasticsearch.ElasticsearchClient;
import co.elastic.clients.elasticsearch._types.mapping.TypeMapping;
import co.elastic.clients.elasticsearch.core.DeleteByQueryRequest;
import co.elastic.clients.json.JsonData;
import co.elastic.clients.util.ObjectBuilder;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.config.LogRepositoryProperties;
import com.elitescloud.boot.log.model.entity.AccessLogEntity;
import com.elitescloud.boot.log.model.entity.LoginLogEntity;
import lombok.extern.log4j.Log4j2;

import java.io.IOException;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.function.Function;

/**
 * Elasticsearch存储.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/18
 */
@Log4j2
public class ElasticsearchStorage extends AbstractLogStorage {

    private final ElasticsearchClient elasticsearchClient;
    private final LogRepositoryProperties.Elasticsearch config;

    public ElasticsearchStorage(LogProperties logProperties, ElasticsearchClient elasticsearchClient) {
        super(logProperties);
        this.elasticsearchClient = elasticsearchClient;
        this.config = logProperties.getRepository().getElasticsearch();
        // 检查索引是否存在
        CompletableFuture.runAsync(this::checkIndexExists)
                .whenComplete((res, e) -> {
                    if (e == null) {
                        log.info("日志持久化方式Elasticsearch初始化成功！");
                        return;
                    }
                    log.error("日志持久化方式Elasticsearch初始化失败：", e);
                });
    }

    @Override
    public void saveAccessLog(AccessLogEntity accessLogEntity) {
        saveLog(indexOfAccessLog(), accessLogEntity, accessLogEntity.getId());
    }

    @Override
    public void saveLoginLog(LoginLogEntity loginLogEntity) {
        saveLog(indexOfLoginLog(), loginLogEntity, loginLogEntity.getId());
    }

    @Override
    public void removeExpiredAccessLog(LocalDateTime expiredTime) {
        removeExpiredLog(indexOfAccessLog(), expiredTime);
    }

    @Override
    public void removeExpiredLoginLog(LocalDateTime expiredTime) {
        removeExpiredLog(indexOfLoginLog(), expiredTime);
    }

    private void saveLog(String indexName, Object logEntity, Long id) {
        try {
            elasticsearchClient.create(builder -> builder
                    .index(indexName)
                    .document(logEntity)
                    .id(id.toString())
            );
        } catch (IOException e) {
            log.error("保存日志异常：", e);
        }
    }

    private void removeExpiredLog(String indexName, LocalDateTime expiredTime) {
        Function<DeleteByQueryRequest.Builder, ObjectBuilder<DeleteByQueryRequest>> deleteRequest = builder ->
                builder.index(indexName)
                        .query(
                                query -> query.range(range -> range.field("requestTime").lte(JsonData.of(expiredTime)))
                        );
        try {
            elasticsearchClient.deleteByQuery(deleteRequest);
        } catch (IOException e) {
            log.error("Elasticsearch删除过期索引异常：", e);
        }
    }

    private void checkIndexExists() {
        Map<String, TypeMapping> indexMap = new HashMap<>(4);
        // 接口访问日志索引
        indexMap.put(indexOfAccessLog(), mappingOfAccessLog());
        // 登录日志素银
        indexMap.put(indexOfLoginLog(), mappingOfLoginLog());

        // 判断是否存在，自动创建索引
        for (var entry : indexMap.entrySet()) {
            try {
                boolean exists = elasticsearchClient.indices()
                        .exists(builder ->
                                builder.index(entry.getKey())
                        ).value();
                if (exists) {
                    // 索引已存在
                    continue;
                }
            } catch (IOException e) {
                throw new IllegalStateException("检查日志索引异常：", e);
            }

            // 是否自动创建
            if (!config.getAutoCreate()) {
                throw new IllegalStateException("日志索引" + entry.getKey() + "不存在，请先创建！");
            }
            try {
                var resp = elasticsearchClient.indices().create(builder ->
                        builder.index(entry.getKey())
                                .mappings(entry.getValue())
                );
                if (!resp.acknowledged()) {
                    throw new IllegalStateException("日志索引" + entry.getKey() + "创建失败");
                }
            } catch (IOException e) {
                throw new RuntimeException("创建日志索引异常：", e);
            }
        }
    }

    private String indexOfAccessLog() {
        return config.getIndexPrefix() + "access";
    }

    private String indexOfLoginLog() {
        return config.getIndexPrefix() + "login";
    }

    private TypeMapping.Builder mappingOfBase() {
        return new TypeMapping.Builder()
                .properties("id", property -> property.long_(p -> p.index(true)))
                .properties("requestTime", property -> property.date(p -> p.index(true).format("uuuu-MM-dd HH:mm:ss")))
                .properties("responseTime", property -> property.date(p -> p.format("uuuu-MM-dd HH:mm:ss")))
                .properties("cost", property -> property.long_(p -> p.index(true)))
                .properties("platformCode", property -> property.keyword(p -> p.index(true)))
                .properties("clientId", property -> property.text(p -> p))
                .properties("userId", property -> property.long_(p -> p))
                .properties("username", property -> property.keyword(p -> p))
                .properties("browser", property -> property.text(p -> p))
                .properties("userAgent", property -> property.text(p -> p))
                .properties("method", property -> property.text(p -> p))
                .properties("reqContentType", property -> property.text(p -> p))
                .properties("uri", property -> property.text(p -> p))
                .properties("operation", property -> property.text(p -> p))
                .properties("reqIp", property -> property.text(p -> p))
                .properties("reqOuterIp", property -> property.text(p -> p))
                .properties("address", property -> property.text(p -> p))
                .properties("queryParams", property -> property.text(p -> p))
                .properties("requestBodyTxt", property -> property.text(p -> p))
                .properties("serverInstance", property -> property.keyword(p -> p.index(true)))
                .properties("serverInstanceIp", property -> property.text(p -> p))
                .properties("appCode", property -> property.keyword(p -> p.index(true)))
                .properties("resultCode", property -> property.integer(p -> p))
                .properties("success", property -> property.boolean_(p -> p))
                .properties("msg", property -> property.text(p -> p))
                .properties("com/elitescloud/cloudt/common/exception", property -> property.text(p -> p))
                ;
    }

    private TypeMapping mappingOfAccessLog() {
        return mappingOfBase()
                .properties("traceId", property -> property.text(p -> p))
                .properties("threadId", property -> property.text(p -> p))
                .properties("responseBodyTxt", property -> property.text(p -> p))
                .build();
    }

    private TypeMapping mappingOfLoginLog() {
        return mappingOfBase()
                .properties("loginMethod", property -> property.text(p -> p))
                .properties("loginType", property -> property.text(p -> p))
                .properties("terminal", property -> property.text(p -> p))
                .properties("userDetail", property -> property.text(p -> p))
                .build();
    }
}
