package com.elitescloud.boot.log.config;

import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.interceptor.AccessLogInterceptor;
import com.elitescloud.boot.log.model.bo.AccessLogBO;
import com.elitescloud.boot.log.common.LogStorable;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Import;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

/**
 * 接口访问日志配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/22
 */
@ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".access-log", name = "enabled", havingValue = "true")
@Import({AccessLogInterceptor.class})
@Log4j2
public class AccessLogConfig implements SchedulingConfigurer {

    private final LogProperties logProperties;
    private LogStorable logStorable;

    public AccessLogConfig(LogProperties logProperties) {
        this.logProperties = logProperties;
        log.info("启用接口访问日志记录");
    }

    @Override
    public void configureTasks(@NonNull ScheduledTaskRegistrar taskRegistrar) {
        var ttl = logProperties.getAccessLog().getTimeToLive();
        if (ttl == null || ttl.toSeconds() < 1) {
            // 未启用自动清理日志
            return;
        }

        // 添加到任务调度
        taskRegistrar.addFixedDelayTask(() -> clearExpiredLog(ttl), TimeUnit.HOURS.toMillis(2));
    }

    @Autowired
    public void setLogStorable(LogStorable logStorable) {
        this.logStorable = logStorable;
    }

    private void clearExpiredLog(Duration ttl) {
        try {
            logStorable.clearExpired(AccessLogBO.class, LocalDateTime.now().minusSeconds(ttl.toSeconds()));
        } catch (Exception e) {
            log.error("清理过期接口访问日志异常：", e);
        }
    }
}
