package com.elitescloud.boot.log.interceptor;

import com.elitescloud.boot.log.common.BusinessObjectAnnotation;
import com.elitescloud.boot.log.common.BusinessOperationAnnotation;
import com.elitescloud.boot.log.operationlog.OperationLogDTO;
import com.elitescloud.boot.log.operationlog.OperationLogMqMessageService;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.aop.aspectj.MethodInvocationProceedingJoinPoint;
import org.springframework.web.bind.annotation.RestController;

import java.time.Duration;
import java.time.Instant;

/**
 * @author : chen.niu
 * @description :
 * @date : 2023-12-11 21:15
 */
@Aspect
@Slf4j
public class BusinessOperationAspect {
    public static final String AUTO = "auto";
    private final OperationLogMqMessageService operationLogService;
    private final ObjectMapper objectMapper = new ObjectMapper();

    public BusinessOperationAspect(OperationLogMqMessageService operationLogService) {
        this.operationLogService = operationLogService;
    }


    @Around("@annotation(businessOperationAnnotation)")
    public Object logBusinessOperation(ProceedingJoinPoint joinPoint, BusinessOperationAnnotation businessOperationAnnotation) throws Throwable {
//        BusinessOperationAnnotation annotation = obtainAnnotation(joinPoint);
        // 检查是否在带有 @RestController 的类的方法上
        if (!joinPoint.getTarget().getClass().isAnnotationPresent(RestController.class)) {
            log.error("注解位置异常：{} BusinessOperationAnnotation 规范应用在  @RestController classes的方法中"
            ,joinPoint.getTarget().getClass().getName());
        }
        if (!joinPoint.getTarget().getClass().isAnnotationPresent(BusinessObjectAnnotation.class)) {
            log.error("注解异常，跳过，继续执行业务代码：{} BusinessOperationAnnotation 规范应用在  @BusinessObjectAnnotation classes的方法中"
            ,joinPoint.getTarget().getClass().getName());
            return joinPoint.proceed();
        }
        if (businessOperationAnnotation.logEnabled()) {
            Instant start = Instant.now();

            String argsJson = "";
            if (businessOperationAnnotation.argsJsonEnabled()) {
                // 转换方法参数为JSON
                argsJson = convertToJson(joinPoint.getArgs());
            }
            // 执行目标方法
            Object result = joinPoint.proceed();
            String resultJson = "";
            if (businessOperationAnnotation.resultJsonEnabled()) {
                // 转换结果为JSON
                resultJson = convertToJson(result);
            }
            BusinessObjectAnnotation businessObjectAnnotation = joinPoint.getTarget().getClass().getAnnotation(BusinessObjectAnnotation.class);
            long timeElapsed = Duration.between(start, Instant.now()).toMillis();

            // 记录操作日志
            logOperation(argsJson, resultJson,timeElapsed, businessObjectAnnotation, businessOperationAnnotation);

            return result;
        } else {
            return joinPoint.proceed();
        }
    }

    @Around("@within(businessObjectAnnotation)")
    public Object checkBusinessObjectAnnotation(ProceedingJoinPoint joinPoint, BusinessObjectAnnotation businessObjectAnnotation) throws Throwable {
        // 检查是否在带有 @RestController 的类上
        if (!joinPoint.getTarget().getClass().isAnnotationPresent(RestController.class)) {
            log.error("注解位置异常：{} BusinessObjectAnnotation 规范应用在 @RestController classes"
                  ,  joinPoint.getTarget().getClass().getName());
        }
        return joinPoint.proceed();
    }


    private void logOperation(String argsJson, String resultJson,long timeElapsed, BusinessObjectAnnotation businessObjectAnnotation, BusinessOperationAnnotation annotation) {
        try {
            OperationLogDTO operationLogDTO = operationLogService.quickNewOperationLogDTO(
                    businessObjectAnnotation.businessType().split(":")[0],
                    annotation.businessParam(),
                    annotation.operationType(),
                    annotation.operationDescription());
            operationLogDTO.setDelayed(timeElapsed);
            operationLogDTO.setOperationAfterData(argsJson);
            operationLogDTO.setOperationBeforeData(resultJson);
            operationLogDTO.setSource(AUTO);
            ApiResult<String> apiResult = operationLogService.sendAsyncOperationLogMqMessage(operationLogDTO);
            if (!apiResult.isSuccess()) {
                log.error("操作日志：消息发送失败 - {}", apiResult.getErrorMsg());
            }
        } catch (Exception e) {
            log.error("操作日志：记录失败 - {}", e.getMessage(), e);
        }
    }


    private String convertToJson(Object object) {
        try {
            return objectMapper.writeValueAsString(object);
        } catch (Exception e) {
            log.error("操作日志：JSON转换异常 - {}", e.getMessage(), e);
            return "转换异常: " + e.getMessage();
        }
    }

    private BusinessOperationAnnotation obtainAnnotation(ProceedingJoinPoint point) {
        if (point instanceof MethodInvocationProceedingJoinPoint) {
            // 优先获取方法上的注解
            BusinessOperationAnnotation annotation = ((MethodSignature) point.getSignature()).getMethod().getAnnotation(BusinessOperationAnnotation.class);
            return annotation;
        }
        return null;
    }
}

