package com.elitescloud.boot.log.interceptor;

/**
 * @author James.Huang
 * @version 2022.1
 * @date 2023/6/26 10:40
 * @Description:
 */

import cn.hutool.extra.servlet.ServletUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.model.bo.OperationLogVO;
import com.elitescloud.boot.util.JSONUtil;
import com.elitescloud.boot.wrapper.CloudtRequestWrapper;
import com.elitescloud.cloudt.security.entity.GeneralUserDetails;
import com.nimbusds.jose.shaded.json.JSONObject;
import lombok.extern.log4j.Log4j2;
import org.apache.commons.lang3.StringUtils;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.core.MethodParameter;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.http.converter.HttpMessageConverter;
import org.springframework.http.server.ServerHttpRequest;
import org.springframework.http.server.ServerHttpResponse;
import org.springframework.web.bind.annotation.ControllerAdvice;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.ModelAttribute;
import org.springframework.web.servlet.mvc.method.annotation.ResponseBodyAdvice;

import javax.servlet.http.HttpServletRequest;
import java.io.ByteArrayOutputStream;
import java.io.PrintStream;
import java.time.LocalDateTime;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;

@Log4j2
@ControllerAdvice
@ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".operation-log", name = "enabled", havingValue = "true")
@ConditionalOnBean(LogProperties.class)
public class OperationLogResponseBodyAdvice implements ResponseBodyAdvice<Object> {

    private final ThreadLocal<OperationLogVO> paramThreadLocal = new ThreadLocal<>();
    private final RedisTemplate redisTemplate;

    public OperationLogResponseBodyAdvice(RedisTemplate redisTemplate) {
        this.redisTemplate = redisTemplate;
    }

    @ModelAttribute
    public void globalAttributes(HttpServletRequest request) {

        String operationConfigKey="OPERATION-CONFIG-KEY";
        String requURI=request.getRequestURI();
        if(requURI==null||StringUtils.isBlank(requURI)){
            return;
        }else if(requURI.contains("/actuator")){
            //系统监控 开头
            return;
        }
        OperationLogVO operationLogVO=new OperationLogVO();
        operationLogVO.setOperationUrl(requURI);
        try{

            boolean isStandkey=redisTemplate.opsForHash().hasKey(operationConfigKey,requURI);
            if(isStandkey){

                String ocObject= (String) redisTemplate.opsForHash().get(operationConfigKey,requURI);
                Map<String,Object> ocMap= JSONUtil.json2Obj(ocObject,HashMap.class);
                if(ocMap.containsKey("sysModel")){
                    operationLogVO.setSysModel(ocMap.get("sysModel").toString());
                }
                if(ocMap.containsKey("triggerTerminal")){
                    operationLogVO.setTriggerTerminal(ocMap.get("triggerTerminal").toString());
                }
                if(ocMap.containsKey("operationType")){
                    operationLogVO.setOperationType(ocMap.get("operationType").toString());
                }

            }
            //操作ip
            operationLogVO.setOperationIp(ServletUtil.getClientIP(request));

            GeneralUserDetails generalUserDetails= SecurityContextUtil.currentUser();

            if(generalUserDetails!=null){
                operationLogVO.setOperationUser(generalUserDetails.getUsername());
            }


            String requestParam="";
            if (request instanceof CloudtRequestWrapper) {
                CloudtRequestWrapper wrapper = (CloudtRequestWrapper) request;
                String body = wrapper.getBodyString();
                if(StringUtils.isNotBlank(body)){
                    String finalBody= removeControlCharacters(body);
                    requestParam="request body param :"+finalBody;
                }
            }
            Map<String, String[]> formMap=request.getParameterMap();
            if(formMap.size()>0){
                if(StringUtils.isNotBlank(requestParam)){
                    requestParam=requestParam+"; request form param: "+JSONObject.toJSONString(request.getParameterMap());
                }else{
                    requestParam="request form param: "+ JSONObject.toJSONString(request.getParameterMap());
                }
            }
            operationLogVO.setRequestParam(requestParam);
            operationLogVO.setOperationTime(LocalDateTime.now());
            String logId= UUID.randomUUID().toString();
            operationLogVO.setLogId(logId);
            operationLogVO.setRequestMethod(request.getMethod());
            request.setAttribute("logId",logId);
            paramThreadLocal.set(operationLogVO);

        }catch (Exception e){
            log.error("获取"+requURI+"信息出错",e);
        }
    }

    public  String removeControlCharacters(String input) {
        if (input == null) {
            return null;
        }
        // 使用正则表达式移除所有控制字符
        return input.replaceAll("\\p{C}", "");
    }

    @Override
    public boolean supports(MethodParameter returnType, Class<? extends HttpMessageConverter<?>> converterType) {
        // 可以根据方法和返回类型判断是否应用此Advice
        return true;
    }

    @ExceptionHandler(Exception.class)
    public ResponseEntity<String> handleException(Exception ex) {
        // 根据异常类型进行处理逻辑，获取API执行状态
        HttpStatus status = HttpStatus.INTERNAL_SERVER_ERROR;
        String message=ex.getMessage();

        if(paramThreadLocal.get()!=null){
            OperationLogVO operationLogVO=paramThreadLocal.get();
            operationLogVO.setOperationStatus("失败");
            operationLogVO.setExceptionDesc(getExceptionAllInfo(ex));

            log.error("执行接口"+operationLogVO.getOperationUrl()+"失败",ex);

            log.error("执行接口"+operationLogVO.getOperationUrl()+"失败,日志信息:"+operationLogVO.toString());
            paramThreadLocal.remove();
        }

        // 构建响应对象
        ResponseEntity<String> response = new ResponseEntity<>(message, status);
        return response;
    }

    public String getExceptionAllInfo(Exception ex) {
        ByteArrayOutputStream out = null;
        PrintStream pout = null;
        String ret = "";
        try {
            out = new ByteArrayOutputStream();
            pout = new PrintStream(out);
            ex.printStackTrace(pout);
            ret = new String(out.toByteArray());
            out.close();
        }catch(Exception e){
            return ex.getMessage();
        }finally{
            if(pout!=null){
                pout.close();
            }
        }
        return ret;
    }

    @Override
    public Object beforeBodyWrite(Object body, MethodParameter returnType, MediaType selectedContentType,
                                  Class<? extends HttpMessageConverter<?>> selectedConverterType,
                                  ServerHttpRequest request, ServerHttpResponse response) {

        if(paramThreadLocal.get()!=null){
            OperationLogVO operationLogVO=paramThreadLocal.get();

            operationLogVO.setOperationStatus("成功");

            if(body!=null){
                operationLogVO.setResponseParam(body.toString());
            }
            log.info("执行API 接口"+operationLogVO.getOperationUrl()+"成功,日志信息:"+operationLogVO.toString());
            paramThreadLocal.remove();
        }
        return body;
    }


}

