package com.elitescloud.boot.log.operationlog;

import cn.hutool.core.date.DateUtil;
import com.elitescloud.boot.auth.util.SecurityContextUtil;
import com.elitescloud.boot.constant.CommonConstant;
import com.elitescloud.cloudt.common.base.ApiResult;
import com.elitescloud.cloudt.context.util.HttpServletUtil;
import lombok.extern.slf4j.Slf4j;
import org.apache.skywalking.apm.toolkit.trace.TraceContext;
import org.slf4j.MDC;
import org.springframework.core.env.Environment;
import org.springframework.core.task.TaskExecutor;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.util.Assert;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.http.HttpServletRequest;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

/**
 * @author : chen.niu
 * @description : 内置操作日志的MQ消息发送。
 * @date : 2023-11-28 17:41
 */
@Slf4j
public class OperationLogMqMessageServiceImpl
        implements OperationLogMqMessageService {

    //人工操作日志标识
    public static final String ARTIFICIAL = "artificial";
    //自定义消息发送
    private final DynamicMessageService messageService;
    private final Environment environment;
    private final TaskExecutor taskExecutor;

    /**操作日志消息固定topic**/
    private final String BINDING_NAME = "YST_OPERATION_LOG_TOPIC";
    private static final String BINDING_PREFIX = "elitesland.log.operation-log.topic.prefix";

    /**读取应用配置appName的参数*/
    private static final String SPRING_APPLICATION_NAME = "spring.application.name";
    /**请求头的菜单编码*/
    private static final String HEADER_MENU_CODE = "menuCode";

    /**接口的编码 暂无用*/
//    private final String HEADER_API_CODE = "apiCode";
    /*约束业务类型入参枚举的命名规范*/
    private static final String BUSINESS_OBJECT_ENUM = "BusinessObjectEnum";
    public OperationLogMqMessageServiceImpl(
            DynamicMessageService messageService,
            Environment environment,
            TaskExecutor taskExecutor
    ) {
        this.messageService = messageService;
        this.environment = environment;
        this.taskExecutor = taskExecutor;
    }

    @Override
    public ApiResult<String> sendSyncOperationLogMqMessage(OperationLogDTO operationLogDTO) {
        try {
            checkOperationLog(operationLogDTO);
            if (sendMessage(operationLogDTO)) {
                return ApiResult.ok();
            } else {
                return ApiResult.fail("发送返回失败状态");
            }

        } catch (Exception e) {
            log.error(e.getMessage());
            return ApiResult.fail(e.getMessage());
        }

    }

    @Override
    public ApiResult<String> sendAsyncOperationLogMqMessage(OperationLogDTO operationLogDTO) {
        try {
            checkOperationLog(operationLogDTO);
            CompletableFuture.runAsync(() -> sendMessage(operationLogDTO), taskExecutor);
            return ApiResult.ok();
        } catch (Exception e) {
            log.error(e.getMessage());
            return ApiResult.fail(e.getMessage());
        }
    }
    String HEADER_KEYS = "KEYS";
    private boolean sendMessage(OperationLogDTO operationLogDTO) {
        String topicPrefix =environment.getProperty(BINDING_PREFIX,"");
        String messageKey= operationLogDTO.getTraceIdSys()+"_"+UUID.randomUUID();
        operationLogDTO.setExt3(topicPrefix+BINDING_NAME+"_"+messageKey);
        var message = MessageBuilder.withPayload(operationLogDTO)
                .setHeader(HEADER_KEYS, messageKey)
                // 可以添加更多的headers
                .build();
        var isSuccess = messageService.sendMessage(
                topicPrefix+BINDING_NAME, operationLogDTO)
                ;
        if (log.isDebugEnabled()) {
            log.debug("操作日志消息发送：{}-{}--{}", isSuccess, operationLogDTO,message);
        }
        return isSuccess;
    }

    @Override
    public OperationLogDTO quickNewOperationLogDTO(
            Enum businessType, String businessParam,
            OperationTypeEnum operationType, String operationDescription) {
        if (!BUSINESS_OBJECT_ENUM.equals(businessType.getClass().getSimpleName())) {
            log.error("命名错误，对象类型枚举统一命名：BusinessObjectEnum");
        }
        return quickNewOperationLogDTO(businessType.name(),businessParam,
                operationType,operationDescription);
    }

    @Override
    public OperationLogDTO quickNewOperationLogDTO(String businessType,
                                                   String businessParam,
                                                   OperationTypeEnum operationType, String operationDescription) {
        /*约束业务类型入参枚举的命名规范*/
        OperationLogDTO operationLogDTO = getOperationLogDTO(businessType, businessParam,
                operationType, operationDescription,environment);
        return operationLogDTO;
    }



    public static OperationLogDTO getOperationLogDTO(String businessType,
                                                     String businessParam,
                                                     OperationTypeEnum operationType,
                                                     String operationDescription,
                                                     Environment environment) {
        OperationLogDTO operationLogDTO = OperationLogDTO.builder().build();
        operationLogDTO.setSource(ARTIFICIAL);
        operationLogDTO.setBusinessParam(businessParam);
        operationLogDTO.setBusinessType(businessType);
        operationLogDTO.setSuccess(true);
        operationLogDTO.setOperationType(operationType.name());
        operationLogDTO.setOperationTypeName(operationType.getDescription());
//        if (!operationLogDTO.getOperationType().equals(OperationTypeEnum.ELSE.name())) {
//            operationLogDTO.setOperationTypeName(operationType.getDescription());
//        }

        operationLogDTO.setOperationDescription(operationDescription);
        operationLogDTO.setAppName(environment.getProperty(SPRING_APPLICATION_NAME));

        HttpServletRequest request = ((ServletRequestAttributes) RequestContextHolder.getRequestAttributes()).getRequest();
        if (request != null) {
            operationLogDTO.setMenuCode(request.getHeader(HEADER_MENU_CODE));
            operationLogDTO.setRequestUrl(request.getRequestURI());
            operationLogDTO.setRequestMethod(request.getMethod());
            operationLogDTO.setOperationIp(HttpServletUtil.currentClientIp());
        } else {
            operationLogDTO.setMenuCode("无菜单");
            operationLogDTO.setRequestUrl("非HTTP请求");
            operationLogDTO.setRequestMethod("非HTTP请求");
            log.info("HttpServletRequest Null");
        }

        operationLogDTO.setToken(SecurityContextUtil.currentToken());
        var user = SecurityContextUtil.currentUser();
        if (user != null) {
            operationLogDTO.setOperationUserName(user.getUsername());
            operationLogDTO.setOperationUserId(user.getUserId());
            operationLogDTO.setTenantId(user.getTenantId()==null?-1L:user.getTenantId());
            operationLogDTO.setOperationName(user.getUser().getLastName());
        } else {
            operationLogDTO.setOperationUserName("无登录人");
            operationLogDTO.setOperationUserId(0L);
            operationLogDTO.setTenantId(-1L);
            operationLogDTO.setOperationName("无登录人");
            log.info("SecurityContextUtil.currentUser Null {}",operationLogDTO);
        }
        operationLogDTO.setOperationTime(DateUtil.now());
        operationLogDTO.setTraceIdSys(MDC.get(CommonConstant.LOG_TRACE_ID));
        operationLogDTO.setTraceIdExt(TraceContext.traceId());
        return operationLogDTO;
    }


    private void checkOperationLog(OperationLogDTO operationLogDTO) {
        Assert.hasText(operationLogDTO.getBusinessType(), "业务类型为空");
        Assert.hasText(operationLogDTO.getBusinessParam(), "业务参数为空");
        Assert.hasText(operationLogDTO.getOperationType(), "操作类型为空");
        Assert.hasText(operationLogDTO.getOperationDescription(), "操作描述为空");
        Assert.hasText(operationLogDTO.getOperationTime(), "操作时间为空");

//        if (operationLogDTO.getOperationType().equals(OperationTypeEnum.ELSE.name())) {
//            Assert.hasText(operationLogDTO.getOperationTypeName(), "其他操作需要自定义说明操作名称 OperationTypeName");
//        }
    }


}
