package com.elitescloud.boot.log.provider.storage;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.model.entity.AccessLogEntity;
import com.elitescloud.boot.log.model.entity.BaseRequestLogEntity;
import com.elitescloud.boot.log.model.entity.LoginLogEntity;
import com.zaxxer.hikari.HikariDataSource;
import lombok.extern.log4j.Log4j2;
import org.springframework.jdbc.core.JdbcTemplate;
import org.springframework.jdbc.core.namedparam.MapSqlParameterSource;
import org.springframework.jdbc.core.namedparam.NamedParameterJdbcTemplate;

import javax.sql.DataSource;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.TimeUnit;

/**
 * 数据库存储.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/18
 */
@Log4j2
public class DatabaseStorage extends AbstractLogStorage {
    private static final String TABLE_ACCESS_LOG = "sys_access_log";
    private static final String TABLE_LOGIN_LOG = "sys_login_log";
    private static final String SQL_INSERT_ACCESS_LOG = "insert into sys_access_log(id, request_time, response_time, " +
            "cost, platform_code, client_id, user_id, username, browser, user_agent, method, req_content_type, uri, " +
            "operation, req_ip, req_outer_ip, address, query_params, request_body_txt, server_instance, server_instance_ip, " +
            "app_code, result_code, success, msg, exception, trace_id, thread_id, response_body_txt) values " +
            "(:id, :request_time, :response_time, :cost, :platform_code, :client_id, :user_id, :username, :browser, " +
            ":user_agent, :method, :req_content_type, :uri, :operation, :req_ip, :req_outer_ip, :address, :query_params, " +
            ":request_body_txt, :server_instance, :server_instance_ip, :app_code, :result_code, :success, :msg, :exception, " +
            ":trace_id, :thread_id, :response_body_txt)";
    private static final String SQL_INSERT_LOGIN_LOG = "insert into sys_login_log(id, request_time, response_time, cost, " +
            "platform_code, client_id, user_id, username, browser, user_agent, method, req_content_type, uri, operation, " +
            "req_ip, req_outer_ip, address, query_params, request_body_txt, server_instance, server_instance_ip, app_code, " +
            "result_code, success, msg, exception, login_method, login_type, terminal, user_detail) values " +
            "(:id, :request_time, :response_time, :cost, :platform_code, :client_id, :user_id, :username, :browser, " +
            ":user_agent, :method, :req_content_type, :uri, :operation, :req_ip, :req_outer_ip, :address, :query_params, " +
            ":request_body_txt, :server_instance, :server_instance_ip, :app_code, :result_code, :success, :msg, :exception, " +
            ":login_method, :login_type, :terminal, :user_detail)";

    private final NamedParameterJdbcTemplate jdbcTemplate;

    private final ArrayBlockingQueue<MapSqlParameterSource> accessLogQueue = new ArrayBlockingQueue<>(204800);

    public DatabaseStorage(LogProperties logProperties) {
        super(logProperties);
        this.jdbcTemplate = new NamedParameterJdbcTemplate(buildJdbcTemplate());

        buildThreadToSaveAccessLog();
    }

    @Override
    public void saveAccessLog(AccessLogEntity accessLogEntity) {
        var param = accessLog2Map(accessLogEntity);
        var result = accessLogQueue.offer(param);
        if (result) {
            return;
        }
        log.error("接口访问日志持久化队列已满！");
    }

    @Override
    public void saveLoginLog(LoginLogEntity loginLogEntity) {
        MapSqlParameterSource params = loginLog2Map(loginLogEntity);
        jdbcTemplate.update(SQL_INSERT_LOGIN_LOG, params);
    }

    @Override
    public void removeExpiredAccessLog(LocalDateTime expiredTime) {
        jdbcTemplate.update("delete from " + TABLE_ACCESS_LOG + " where request_time < :expiredTime", Map.of("expiredTime", expiredTime));
    }

    @Override
    public void removeExpiredLoginLog(LocalDateTime expiredTime) {
        jdbcTemplate.update("delete from " + TABLE_LOGIN_LOG + " where request_time < :expiredTime", Map.of("expiredTime", expiredTime));
    }

    private JdbcTemplate buildJdbcTemplate() {
        return new JdbcTemplate(buildDatasource());
    }

    private DataSource buildDatasource() {
        var database = logProperties.getRepository().getDatabase();

        HikariDataSource dataSource = new HikariDataSource();
        dataSource.setJdbcUrl(Assert.notBlank(database.getUrl(), "日志配置的数据库url为空"));
        dataSource.setDriverClassName(Assert.notBlank(database.getDriverClassName(), "日志配置的数据库驱动为空"));
        dataSource.setUsername(Assert.notBlank(database.getUsername(), "日志配置的数据库用户名为空"));
        dataSource.setPassword(Assert.notBlank(database.getPassword(), "日志配置的数据库密码为空"));
        return dataSource;
    }

    private MapSqlParameterSource accessLog2Map(AccessLogEntity entity) {
        MapSqlParameterSource parameterSource = new MapSqlParameterSource();
        fillBaseRequestLogEntity(parameterSource, entity);

        parameterSource.addValue("trace_id", entity.getTraceId());
        parameterSource.addValue("thread_id", entity.getThreadId());
        parameterSource.addValue("response_body_txt", entity.getResponseBodyTxt());

        return parameterSource;
    }

    private MapSqlParameterSource loginLog2Map(LoginLogEntity entity) {
        MapSqlParameterSource parameterSource = new MapSqlParameterSource();
        fillBaseRequestLogEntity(parameterSource, entity);

        parameterSource.addValue("login_method", entity.getLoginMethod());
        parameterSource.addValue("login_type", entity.getLoginType());
        parameterSource.addValue("terminal", entity.getTerminal());
        parameterSource.addValue("user_detail", entity.getUserDetail());

        return parameterSource;
    }

    private void fillBaseRequestLogEntity(MapSqlParameterSource parameterSource, BaseRequestLogEntity entity) {
        parameterSource.addValue("id", entity.getId());
        parameterSource.addValue("request_time", entity.getRequestTime());
        parameterSource.addValue("response_time", entity.getResponseTime());
        parameterSource.addValue("cost", entity.getCost());
        parameterSource.addValue("platform_code", entity.getPlatformCode());
        parameterSource.addValue("client_id", entity.getClientId());
        parameterSource.addValue("user_id", entity.getUserId());
        parameterSource.addValue("username", entity.getUsername());
        parameterSource.addValue("browser", entity.getBrowser());
        parameterSource.addValue("user_agent", entity.getUserAgent());
        parameterSource.addValue("method", entity.getMethod());
        parameterSource.addValue("req_content_type", entity.getReqContentType());
        parameterSource.addValue("uri", entity.getUri());
        parameterSource.addValue("operation", entity.getOperation());
        parameterSource.addValue("req_ip", entity.getReqIp());
        parameterSource.addValue("req_outer_ip", entity.getReqOuterIp());
        parameterSource.addValue("address", entity.getAddress());
        parameterSource.addValue("query_params", entity.getQueryParams());
        parameterSource.addValue("request_body_txt", entity.getRequestBodyTxt());
        parameterSource.addValue("server_instance", entity.getServerInstance());
        parameterSource.addValue("server_instance_ip", entity.getServerInstanceIp());
        parameterSource.addValue("app_code", entity.getAppCode());
        parameterSource.addValue("result_code", entity.getResultCode());
        parameterSource.addValue("success", entity.getSuccess());
        parameterSource.addValue("msg", entity.getMsg());
        parameterSource.addValue("com/elitescloud/cloudt/common/exception", entity.getException());
    }

    /**
     * 创建线程负责持久化访问日志
     */
    @SuppressWarnings("unchecked")
    private void buildThreadToSaveAccessLog() {
        var thread = new Thread(() -> {
            while (true) {
                try {
                    // 持久化
                    executeSaveAccessLog();
                    TimeUnit.SECONDS.sleep(logProperties.getAccessLog().getPersistenceRate().toSeconds());
                } catch (Exception e) {
                    log.error("持久化接口日志线程异常：", e);
                }
            }
        });
        thread.setDaemon(true);
        thread.setName("cloudt-accessLog-persistence");
        thread.start();
    }

    private void executeSaveAccessLog() {
        // 每次持久化最多限制的数量
        int max = ObjectUtil.defaultIfNull(logProperties.getAccessLog().getMax(), 1024);
        List<MapSqlParameterSource> sourceList = new ArrayList<>(max);

        int size = 0;
        MapSqlParameterSource element = null;
        while (size < max || !accessLogQueue.isEmpty()) {
            element = accessLogQueue.poll();
            if (element == null) {
                // 队列空了
                break;
            }
            sourceList.add(element);
            size++;
        }
        if (!sourceList.isEmpty()) {
            jdbcTemplate.batchUpdate(SQL_INSERT_ACCESS_LOG, sourceList.toArray(MapSqlParameterSource[]::new));
        }
    }
}
