package com.elitescloud.boot.log;

import com.elitescloud.boot.log.config.LogRepositoryProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.NestedConfigurationProperty;
import org.springframework.validation.annotation.Validated;

import javax.validation.Valid;
import javax.validation.constraints.NotBlank;
import javax.validation.constraints.Pattern;
import java.time.Duration;
import java.util.List;

/**
 * 日志相关属性配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/16
 */
@ConfigurationProperties(prefix = LogProperties.CONFIG_PREFIX)
public class LogProperties {
    public static final String CONFIG_PREFIX = "elitesland.log";

    /**
     * 是否启用日志功能
     */
    private Boolean enabled = true;

    /**
     * 日志处理线程池
     */
    private final ThreadPool threadPool = new ThreadPool();

    /**
     * 日志持久化配置
     */
    @NestedConfigurationProperty
    private LogRepositoryProperties repository = new LogRepositoryProperties();

    /**
     * 接口访问日志
     */
    @Valid
    @NestedConfigurationProperty
    private AccessLog accessLog = new AccessLog();

    /**
     * 登录日志
     */
    @NestedConfigurationProperty
    private LoginLog loginLog = new LoginLog();

    /**
     * 操作日志
     */
    @NestedConfigurationProperty
    private OperationLog operationLog = new OperationLog();

    public Boolean getEnabled() {
        return enabled;
    }

    public void setEnabled(Boolean enabled) {
        this.enabled = enabled;
    }

    public ThreadPool getThreadPool() {
        return threadPool;
    }

    public void setRepository(LogRepositoryProperties repository) {
        this.repository = repository;
    }

    public LogRepositoryProperties getRepository() {
        return repository;
    }

    public AccessLog getAccessLog() {
        return accessLog;
    }

    public LoginLog getLoginLog() {
        return loginLog;
    }

    public void setAccessLog(AccessLog accessLog) {
        this.accessLog = accessLog;
    }

    public void setLoginLog(LoginLog loginLog) {
        this.loginLog = loginLog;
    }

    public OperationLog getOperationLog() {
        return operationLog;
    }

    public void setOperationLog(OperationLog operationLog) {
        this.operationLog = operationLog;
    }

    public static class AccessLog {
        /**
         * 是否开启
         */
        private Boolean enabled = false;

        /**
         * 不做记录的请求
         */
        @Valid
        private List<Matcher> excludedRequest;

        /**
         * 记录响应体的接口
         */
        @Valid
        private List<Matcher> recordResponseBody;

        /**
         * 记录请求体的接口
         */
        @Valid
        private List<Matcher> recordRequestBody;

        /**
         * 批量持久化时最大的数量
         */
        private Integer max = 512;

        /**
         * 持久化频率，默认20秒
         */
        private Duration persistenceRate = Duration.ofSeconds(20);

        /**
         * 日志保留时间，默认30天
         */
        private Duration timeToLive = Duration.ofDays(30);

        public Boolean getEnabled() {
            return enabled;
        }

        public void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public List<Matcher> getExcludedRequest() {
            return excludedRequest;
        }

        public void setExcludedRequest(List<Matcher> excludedRequest) {
            this.excludedRequest = excludedRequest;
        }

        public List<Matcher> getRecordResponseBody() {
            return recordResponseBody;
        }

        public void setRecordResponseBody(List<Matcher> recordResponseBody) {
            this.recordResponseBody = recordResponseBody;
        }

        public List<Matcher> getRecordRequestBody() {
            return recordRequestBody;
        }

        public void setRecordRequestBody(List<Matcher> recordRequestBody) {
            this.recordRequestBody = recordRequestBody;
        }

        public Integer getMax() {
            return max;
        }

        public void setMax(Integer max) {
            this.max = max;
        }

        public Duration getPersistenceRate() {
            return persistenceRate;
        }

        public void setPersistenceRate(Duration persistenceRate) {
            this.persistenceRate = persistenceRate;
        }

        public Duration getTimeToLive() {
            return timeToLive;
        }

        public void setTimeToLive(Duration timeToLive) {
            this.timeToLive = timeToLive;
        }
    }

    public static class LoginLog {
        /**
         * 是否开启
         */
        private Boolean enabled = true;

        /**
         * 是否记录用户详细信息
         */
        private Boolean userDetail = false;

        /**
         * 日志保留时间
         */
        private Duration timeToLive = Duration.ofDays(180);

        public Boolean getEnabled() {
            return enabled;
        }

        public void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        public Boolean getUserDetail() {
            return userDetail;
        }

        public void setUserDetail(Boolean userDetail) {
            this.userDetail = userDetail;
        }

        public Duration getTimeToLive() {
            return timeToLive;
        }

        public void setTimeToLive(Duration timeToLive) {
            this.timeToLive = timeToLive;
        }
    }

    public static class OperationLog {
        /**
         * 是否开启
         */
        private Boolean enabled = true;

        public Boolean getEnabled() {
            return enabled;
        }

        public void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }
    }

    @Validated
    public static class Matcher {
        /**
         * uri路径
         */
        @NotBlank(message = "uri不能为空")
        @Pattern(regexp = "/[^?#]*", message = "uri必须以/开始")
        private String uri;

        public String getUri() {
            return uri;
        }

        public void setUri(String uri) {
            this.uri = uri;
        }
    }

    public static class ThreadPool {
        /**
         * 线程名称前缀
         */
        private String threadNamePrefix = "cloudt-log-";

        /**
         * 最小线程数量
         */
        private Integer coreSize = 8;

        /**
         * 最多线程池数量
         */
        private Integer maxSize = 64;

        public String getThreadNamePrefix() {
            return threadNamePrefix;
        }

        public void setThreadNamePrefix(String threadNamePrefix) {
            this.threadNamePrefix = threadNamePrefix;
        }

        public Integer getCoreSize() {
            return coreSize;
        }

        public void setCoreSize(Integer coreSize) {
            this.coreSize = coreSize;
        }

        public Integer getMaxSize() {
            return maxSize;
        }

        public void setMaxSize(Integer maxSize) {
            this.maxSize = maxSize;
        }
    }
}
