package com.elitescloud.boot.log.config;

import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.model.bo.LoginLogBO;
import com.elitescloud.boot.log.common.LogStorable;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.lang.NonNull;
import org.springframework.scheduling.annotation.SchedulingConfigurer;
import org.springframework.scheduling.config.ScheduledTaskRegistrar;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

/**
 * 登录日志配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/22
 */
@ConditionalOnProperty(prefix = LogProperties.CONFIG_PREFIX + ".login-log", name = "enabled", havingValue = "true", matchIfMissing = true)
@Log4j2
public class LoginLogConfig implements SchedulingConfigurer {

    private final LogProperties logProperties;
    private LogStorable logStorable;

    public LoginLogConfig(LogProperties logProperties) {
        this.logProperties = logProperties;
        log.info("启用登录日志记录");
    }

    @Override
    public void configureTasks(@NonNull ScheduledTaskRegistrar taskRegistrar) {
        var ttl = logProperties.getLoginLog().getTimeToLive();
        if (ttl == null || ttl.toSeconds() < 1) {
            // 未启用自动清理日志
            return;
        }

        // 添加到任务调度
        taskRegistrar.addFixedDelayTask(() -> clearExpiredLog(ttl), TimeUnit.HOURS.toMillis(2));
    }

    @Autowired
    public void setLogStorable(LogStorable logStorable) {
        this.logStorable = logStorable;
    }

    private void clearExpiredLog(Duration ttl) {
        try {
            logStorable.clearExpired(LoginLogBO.class, LocalDateTime.now().minusSeconds(ttl.toSeconds()));
        } catch (Exception e) {
            log.error("清理过期登录日志异常：", e);
        }
    }
}
