package com.elitescloud.boot.log.provider.storage;

import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.http.useragent.UserAgentUtil;
import com.elitescloud.boot.constant.AuthenticationClaim;
import com.elitescloud.boot.log.LogProperties;
import com.elitescloud.boot.log.convert.LogConvert;
import com.elitescloud.boot.log.model.bo.AccessLogBO;
import com.elitescloud.boot.log.model.bo.BaseRequestLogBO;
import com.elitescloud.boot.log.model.bo.LoginLogBO;
import com.elitescloud.boot.log.model.entity.AccessLogEntity;
import com.elitescloud.boot.log.model.entity.BaseRequestLogEntity;
import com.elitescloud.boot.log.model.entity.LoginLogEntity;
import com.elitescloud.boot.log.common.LogStorable;
import com.elitescloud.boot.support.app.CloudtAppHolder;
import com.elitescloud.boot.provider.IdFactory;
import com.elitescloud.boot.util.JwtUtil;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateSerializer;
import com.fasterxml.jackson.datatype.jsr310.ser.LocalDateTimeSerializer;
import lombok.extern.log4j.Log4j2;
import org.springframework.http.converter.json.Jackson2ObjectMapperBuilder;
import org.springframework.util.Assert;
import org.springframework.util.CollectionUtils;
import org.springframework.util.StringUtils;

import java.time.Duration;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Collections;
import java.util.Map;

/**
 * 日志存储的父类.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/18
 */
@Log4j2
public abstract class AbstractLogStorage implements LogStorable {

    protected final ObjectMapper objectMapper;
    protected static final LogConvert CONVERT = LogConvert.INSTANCE;

    protected final LogProperties logProperties;

    protected AbstractLogStorage(LogProperties logProperties) {
        this.logProperties = logProperties;
        objectMapper = buildObjectMapper();
    }

    /**
     * 保存接口访问日志
     *
     * @param accessLogEntity 接口访问日志
     */
    public abstract void saveAccessLog(AccessLogEntity accessLogEntity);

    /**
     * 保存登录日志
     *
     * @param loginLogEntity 登录日志
     */
    public abstract void saveLoginLog(LoginLogEntity loginLogEntity);

    /**
     * 清理过期的接口访问日志
     *
     * @param expiredTime 过期时间
     */
    public abstract void removeExpiredAccessLog(LocalDateTime expiredTime);

    /**
     * 清理过期的登录日志
     *
     * @param expiredTime 过期时间
     */
    public abstract void removeExpiredLoginLog(LocalDateTime expiredTime);

    @Override
    public void storage(Object log) {
        if (log == null) {
            return;
        }

        if (log instanceof AccessLogBO) {
            // 接口访问日志
            var entity = toEntity((AccessLogBO) log);
            saveAccessLog(entity);
        } else if (log instanceof LoginLogBO) {
            // 登录日志
            var entity = toEntity((LoginLogBO) log);
            saveLoginLog(entity);
        }
    }

    @Override
    public void clearExpired(Class<?> logType, LocalDateTime expiredTime) {
        Assert.notNull(expiredTime, "过期时间为空");

        if (logType == AccessLogBO.class) {
            removeExpiredAccessLog(expiredTime);
        } else if (logType == LoginLogBO.class) {
            removeExpiredLoginLog(expiredTime);
        }
    }

    /**
     * 生成ID
     *
     * @return id
     */
    protected Long generateId() {
        return IdFactory.generateLong();
    }

    /**
     * 对象转json字符串
     *
     * @param obj 对象
     * @return json字符串
     */
    protected String obj2Json(Object obj) {
        if (obj == null) {
            return null;
        }

        if (obj.getClass().isPrimitive()) {
            return obj.toString();
        }

        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            log.error("转换JSON字符串异常：", e);
            return null;
        }
    }

    private AccessLogEntity toEntity(AccessLogBO logBO) {
        var entity = CONVERT.toEntity(logBO);
        fillBaseEntity(entity, logBO);

        entity.setResponseBodyTxt(obj2Json(logBO.getResult()));

        return entity;
    }

    private LoginLogEntity toEntity(LoginLogBO logBO) {
        var entity = CONVERT.toEntity(logBO);
        fillBaseEntity(entity, logBO);

        entity.setUserDetail(obj2Json(logBO.getUserDetailObj()));

        return entity;
    }

    private <T extends BaseRequestLogEntity, S extends BaseRequestLogBO> void fillBaseEntity(T entity, S bo) {
        entity.setId(generateId());
        entity.setCost(Duration.between(entity.getRequestTime(), entity.getResponseTime()).toMillis());

        // 从token解析认证相关信息
        Map<String, Object> claims = decodeToken(bo.getToken());
        if (!claims.isEmpty()) {
            entity.setPlatformCode((String) claims.get(AuthenticationClaim.KEY_PLATFORM_CODE));
            entity.setClientId((String) claims.get(AuthenticationClaim.KEY_CLIENT_ID));
            if (claims.containsKey(AuthenticationClaim.KEY_USERID)) {
                entity.setUserId(Long.parseLong(claims.get(AuthenticationClaim.KEY_USERID).toString()));
            }
            entity.setUsername((String) claims.get(AuthenticationClaim.KEY_USERNAME));
        }

        // 浏览器标识
        var userAgent = UserAgentUtil.parse(entity.getUserAgent());
        if (userAgent != null) {
            entity.setBrowser(userAgent.getBrowser().getName());
        }

        // 请求体
        var requestBodies = bo.getRequestBody();
        if (!CollectionUtils.isEmpty(requestBodies)) {
            entity.setRequestBodyTxt(requestBodies.size() == 1 ? obj2Json(requestBodies.get(0)) : obj2Json(requestBodies));
        }

        // 服务相关信息
        entity.setServerInstance(CloudtAppHolder.getServerInstance());
        entity.setServerInstanceIp(CloudtAppHolder.getServerIp());
        entity.setAppCode(CloudtAppHolder.getAppCode());

        // 操作结果
        Integer resultCode = entity.getResultCode();
        entity.setSuccess(resultCode != null && (resultCode >= 200 && resultCode < 300));

        if (bo.getThrowable() != null) {
            var limit = logProperties.getRepository().getExceptionStacktraceLength();
            entity.setException(ExceptionUtil.stacktraceToString(bo.getThrowable(), limit));
        }
    }

    private Map<String, Object> decodeToken(String token) {
        if (!StringUtils.hasText(token)) {
            return Collections.emptyMap();
        }

        try {
            return JwtUtil.decode(token);
        } catch (Exception e) {
            log.warn("解密token失败：", e);
        }
        return Collections.emptyMap();
    }

    private ObjectMapper buildObjectMapper() {
        return new Jackson2ObjectMapperBuilder()
                .simpleDateFormat("yyyy-MM-dd HH:mm:ss.SSS")
                .serializerByType(LocalDateTime.class, new LocalDateTimeSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd HH:mm:ss.SSS")))
                .serializerByType(LocalDate.class, new LocalDateSerializer(DateTimeFormatter.ofPattern("yyyy-MM-dd")))
                .build();
    }
}
