package com.elitescloud.cloudt.messenger.config.support;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.elitescloud.cloudt.common.base.RpcResult;
import com.elitescloud.cloudt.messenger.common.MessengerUriConstants;
import com.elitescloud.cloudt.messenger.config.CloudtMessengerProperties;
import com.elitescloud.cloudt.messenger.config.MessengerSenderSupport;
import com.elitescloud.cloudt.messenger.message.EmailMessageVO;
import com.elitescloud.cloudt.messenger.message.SiteMessageVO;
import com.elitescloud.cloudt.messenger.message.SmsMessageVO;
import com.elitescloud.cloudt.messenger.model.AbstractMessageVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.Set;

/**
 * 消息发送提供者.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/22
 */
public class CloudtMessengerSender {
    private static final Logger LOG = LoggerFactory.getLogger(CloudtMessengerSender.class);
    private static CloudtMessengerSender messangerSender;

    private final RestTemplateHelper restTemplateHelper;

    private CloudtMessengerSender(CloudtMessengerProperties messengerProperties) {
        this.restTemplateHelper = RestTemplateHelper.instance(RestTemplateFactory.instance(builder ->
                builder.rootUri(messengerProperties.getServerUrl())
                        .additionalInterceptors(new AuthorizationInterceptor(messengerProperties))
        ));
    }

    /**
     * 创建实例
     *
     * @return 发送实例
     */
    public static CloudtMessengerSender instance() {
        if (messangerSender == null) {
            return buildInstance();
        }
        return messangerSender;
    }

    /**
     * 发送消息
     *
     * @param messenger 消息对象
     * @param <T>       消息类型
     * @return 消息回执编码
     */
    public <T extends AbstractMessageVO> String send(T messenger) {
        var uri = this.detectSendUri(messenger);

        RpcResult<String> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.POST, new HttpEntity<>(messenger), new ParameterizedTypeReference<>() {
            });
        } catch (Exception e) {
            LOG.error("发送消息异常：", e);
            throw e;
        }

        LOG.info("发送消息{}结果：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        if (rpcResult.isSuccess()) {
            return rpcResult.getData();
        }
        return null;
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageId 消息ID
     * @return 消息状态
     */
    public String getMsgStatus(@NotBlank String messageId) {
        Assert.notBlank(messageId, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_STATUS + "?messageId={messageId}";
        RpcResult<String> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            }, messageId);
        } catch (Exception e) {
            LOG.error("查询消息状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return rpcResult.getData();
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageIds 消息ID
     * @return ID与消息状态
     */
    public Map<String, String> queryMsgStatus(@NotEmpty Set<String> messageIds) {
        Assert.notEmpty(messageIds, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_STATUS_BATCH;
        RpcResult<Map<String, String>> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.POST, new HttpEntity<>(messageIds), new ParameterizedTypeReference<>() {
            });
        } catch (Exception e) {
            LOG.error("查询消息状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return ObjectUtil.defaultIfNull(rpcResult.getData(), Collections.emptyMap());
    }

    /**
     * 获取消息的接收状态
     *
     * @param messageId 消息ID
     * @return 账号与接收状态
     */
    public Map<String, String> queryMsgReceiveStatus(@NotBlank String messageId) {
        Assert.notBlank(messageId, "消息ID为空");
        var uri = MessengerUriConstants.MSG_RECEIVE_STATUS + "?messageId={messageId}";
        RpcResult<Map<String, String>> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            }, messageId);
        } catch (Exception e) {
            LOG.error("查询消息接收状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}接收状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return ObjectUtil.defaultIfNull(rpcResult.getData(), Collections.emptyMap());
    }

    private <T extends AbstractMessageVO> String detectSendUri(T messenger) {
        if (messenger instanceof SmsMessageVO) {
            return MessengerUriConstants.SENDER_SMS;
        }
        if (messenger instanceof EmailMessageVO) {
            return MessengerUriConstants.SENDER_EMAIL;
        }
        if (messenger instanceof SiteMessageVO) {
            return MessengerUriConstants.SENDER_SITE;
        }
        throw new IllegalStateException("发送失败，暂不支持的消息类型");
    }

    private static CloudtMessengerSender buildInstance() {
        Assert.state(MessengerSenderSupport.getMessengerProperties() != null, "消息发送初始化失败");
        synchronized (CloudtMessengerSender.class) {
            if (messangerSender != null) {
                return messangerSender;
            }

            // 初始化
            messangerSender = new CloudtMessengerSender(MessengerSenderSupport.getMessengerProperties());

            return messangerSender;
        }
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        private final CloudtMessengerProperties messengerProperties;

        public AuthorizationInterceptor(CloudtMessengerProperties messengerProperties) {
            this.messengerProperties = messengerProperties;
        }

        @Override
        public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
            if (Boolean.TRUE.equals(messengerProperties.getAuthorization())) {
                var requestAttributes = RequestContextHolder.getRequestAttributes();
                if (requestAttributes != null) {
                    var originalRequest = ((ServletRequestAttributes) requestAttributes).getRequest();
                    var token = originalRequest.getHeader(HttpHeaders.AUTHORIZATION);
                    if (StringUtils.hasText(token)) {
                        request.getHeaders().add(HttpHeaders.AUTHORIZATION, token);
                    }
                }
            }
            return execution.execute(request, body);
        }
    }
}
