package com.elitescloud.cloudt.messenger.message;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ArrayUtil;
import com.elitescloud.boot.util.ObjectMapperFactory;
import com.elitescloud.cloudt.messenger.config.MessengerSenderSupport;
import com.elitescloud.cloudt.messenger.config.support.CloudtMessengerSender;
import com.elitescloud.cloudt.messenger.model.AbstractMessageVO;
import com.elitescloud.cloudt.messenger.model.ByteMessageAttachmentVO;
import com.elitescloud.cloudt.messenger.model.FileMessageAttachmentVO;
import com.elitescloud.cloudt.messenger.model.MessageAttachmentVO;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.Data;

import java.util.Map;

/**
 * 消息父类.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/22
 */
@Data
public abstract class AbstractMessageSenderVO extends AbstractMessageVO {
    private static final long serialVersionUID = 270537922564251133L;
    private static final ObjectMapper OBJECT_MAPPER = ObjectMapperFactory.instance();

    /**
     * 发送前的校验
     */
    protected void validate() {
    }

    /**
     * 发送消息
     *
     * @return 消息回执编号
     */
    public String send() {
        return this.send(false);
    }

    /**
     * 同步发送消息
     *
     * @return 消息回执编号
     */
    public String sendSync() {
        return this.send(true);
    }

    protected final void validateAttachment(MessageAttachmentVO attachmentVO) {
        if (attachmentVO instanceof ByteMessageAttachmentVO) {
            var att = (ByteMessageAttachmentVO) attachmentVO;
            Assert.isTrue(ArrayUtil.isNotEmpty(att.getFileBytes()), "附件的字节数组为空");
            Assert.notBlank(att.getShowName(), "文件名称不能为空");
            return;
        }
        if (attachmentVO instanceof FileMessageAttachmentVO) {
            var att = (FileMessageAttachmentVO) attachmentVO;
            Assert.notBlank(att.getFileCode(), "附件的文件编码为空");
            return;
        }
        throw new IllegalArgumentException("暂不支持的附件类型：" + attachmentVO.getClass().getName());
    }

    /**
     * 发送消息
     *
     * @param sync 是否同步发送
     * @return 消息回执编号
     */
    private String send(boolean sync) {
        // 设置是否同步
        super.setSync(sync);
        // 设置消息渠道
        super.setChannelCode(this.detectChannel());
        // 设置发送配置
        super.setProperties(this.detectConfigProperties());
        // 设置操作人
        super.setCreator(this.obtainCreator());

        // 自校验
        this.validate();

        return CloudtMessengerSender.instance().send(this);
    }

    private String obtainCreator() {
        var currentUserProvider = MessengerSenderSupport.getCurrentUserProvider();
        if (currentUserProvider == null) {
            return null;
        }
        return currentUserProvider.currentUsername();
    }

    /**
     * 选择渠道
     *
     * @return 消息渠道
     */
    private String detectChannel() {
        var chooseService = MessengerSenderSupport.getMessageChannelChoose();
        if (chooseService == null) {
            return null;
        }

        return chooseService.choose(this);
    }

    /**
     * 查询属性配置
     *
     * @return 属性配置
     */
    private Map<String, Object> detectConfigProperties() {
        var chooseService = MessengerSenderSupport.getMessagePropertiesChoose();
        if (chooseService == null) {
            return null;
        }

        // 选择配置
        var prop = chooseService.choose(this);
        if (prop == null) {
            return null;
        }

        return OBJECT_MAPPER.convertValue(prop, new TypeReference<>() {
        });
    }
}
