package com.elitescloud.cloudt.messenger;

import com.elitescloud.cloudt.messenger.common.MsgReceiveStatusEnum;
import com.elitescloud.cloudt.messenger.common.MsgSendStatusEnum;
import com.elitescloud.cloudt.messenger.config.support.CloudtMessengerSender;
import com.elitescloud.cloudt.messenger.message.builder.*;
import com.elitescloud.cloudt.messenger.model.dto.MsgResultDTO;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

/**
 * 消息构建发送.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/19
 */
public class Messenger {
    private Messenger() {
    }

    /**
     * email消息
     * <p>
     * 支持普通文本邮件信息和附件
     *
     * @return email消息构建器
     */
    public static EmailMessageBuilder email() {
        return new EmailMessageBuilder();
    }

    /**
     * 富文本email消息
     * <p>
     * 支持html内容、内联文件和附件
     *
     * @return 富文本email消息构建器
     */
    public static RichTextEmailMessageBuilder richTextEmail() {
        return new RichTextEmailMessageBuilder();
    }

    /**
     * 短信
     *
     * @return 短信构建器
     */
    public static SmsMessageBuilder sms() {
        return new SmsMessageBuilder();
    }

    /**
     * 模板短信
     *
     * @return 模板短信构建器
     */
    public static TemplateSmsMessageBuilder templateSms() {
        return new TemplateSmsMessageBuilder();
    }

    /**
     * 站内信
     *
     * @return 站内信构建器
     */
    public static SiteMessageBuilder siteMessage() {
        return new SiteMessageBuilder();
    }

    /**
     * App消息
     *
     * @return App消息构建器
     */
    public static AppMessageBuilder appMessage() {
        return new AppMessageBuilder();
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageId 消息ID
     * @return 发送状态 {@link MsgSendStatusEnum}
     */
    public static String querySendStatus(@NotBlank String messageId) {
        return CloudtMessengerSender.instance().getMsgStatus(messageId);
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageIds 消息ID
     * @return 消息ID与发送状态 {@link MsgSendStatusEnum} 的映射
     */
    public static Map<String, String> querySendStatus(@NotEmpty Set<String> messageIds) {
        return CloudtMessengerSender.instance().queryMsgStatus(messageIds);
    }

    /**
     * 查询消息的接收状态
     *
     * @param messageId 消息ID
     * @return 接收账号与接收状态 {@link MsgReceiveStatusEnum} 的映射
     */
    public static Map<String, String> queryReceiveStatus(@NotBlank String messageId) {
        return CloudtMessengerSender.instance().queryMsgReceiveStatus(messageId);
    }

    /**
     * 查询消息的接收状态
     *
     * @param messageId 消息ID
     * @return 接收账号与接收状态 {@link MsgReceiveStatusEnum} 的映射
     */
    public static Boolean updateReceiveStatus(@NotBlank String messageId, @NotBlank String receiver, @NotNull MsgReceiveStatusEnum receiveStatus) {
        return CloudtMessengerSender.instance().updateMsgReceiveStatus(messageId, receiver, receiveStatus);
    }

    /**
     * 重试发送消息
     *
     * @param messageId 消息ID
     * @param sync      是否同步发送
     * @return 消息ID
     */
    public static String retrySend(@NotBlank String messageId, Boolean sync) {
        return CloudtMessengerSender.instance().retryMsg(messageId, sync);
    }

    /**
     * 查询消息的发送结果
     *
     * @param messageId 消息ID
     * @return 发送结果
     */
    public static MsgResultDTO querySendResult(@NotBlank String messageId) {
        return CloudtMessengerSender.instance().querySendResult(messageId);
    }

    /**
     * 查询消息的发送结果
     *
     * @param messageIds 消息ID
     * @return 发送结果
     */
    public static List<MsgResultDTO> querySendResult(@NotEmpty Collection<String> messageIds) {
        return CloudtMessengerSender.instance().querySendResult(messageIds);
    }


}
