package com.elitescloud.cloudt.messenger.config;

import com.elitescloud.cloudt.messenger.common.MessengerChannelConstant;
import com.elitescloud.cloudt.messenger.config.support.CurrentUserProvider;
import com.elitescloud.cloudt.messenger.config.support.MessageChannelChoose;
import com.elitescloud.cloudt.messenger.config.support.MessagePropertiesChoose;
import com.elitescloud.cloudt.messenger.message.AppMessageVO;
import com.elitescloud.cloudt.messenger.message.EmailMessageVO;
import com.elitescloud.cloudt.messenger.message.SiteMessageVO;
import com.elitescloud.cloudt.messenger.message.SmsMessageVO;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.security.core.context.SecurityContext;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.util.StringUtils;

/**
 * 消息发送的自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/22
 */
@Configuration(proxyBeanMethods = false)
@ConditionalOnProperty(prefix = CloudtMessengerProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(CloudtMessengerProperties.class)
class CloudtMessengerAutoConfiguration implements InitializingBean, ApplicationContextAware {
    private static final Logger LOG = LoggerFactory.getLogger(CloudtMessengerAutoConfiguration.class);

    private final CloudtMessengerProperties messengerProperties;
    private ApplicationContext applicationContext;

    public CloudtMessengerAutoConfiguration(CloudtMessengerProperties messengerProperties) {
        this.messengerProperties = messengerProperties;
        LOG.info("消息服务：{}", messengerProperties.getServerUrl());
    }

    @Bean
    @ConditionalOnMissingBean
    public MessagePropertiesChoose messagePropertiesChooseDefault() {
        return messageVO -> {
            if (messageVO instanceof EmailMessageVO) {
                // 邮件信息
                return messengerProperties.getSender().getEmail();
            } else if (messageVO instanceof SmsMessageVO) {
                // 短信消息
                if (StringUtils.hasText(messengerProperties.getSender().getAliyunSms().getAccessKeyId())) {
                    return messengerProperties.getSender().getAliyunSms();
                }
            } else if (messageVO instanceof SiteMessageVO) {
                // 站内信
                return messengerProperties.getSender().getSiteMessage();
            } else if (messageVO instanceof AppMessageVO) {
                // App消息
                if (StringUtils.hasText(messengerProperties.getSender().getJpush().getAppKey())) {
                    return messengerProperties.getSender().getJpush();
                }
            } else {
                LOG.warn("获取消息属性配置异常，不支持的消息类型：{}", messageVO.getClass().getName());
            }
            return null;
        };
    }

    @Bean
    @ConditionalOnMissingBean
    public MessageChannelChoose messageChannelChooseDefault() {
        return messageVO -> {
            if (messageVO instanceof EmailMessageVO) {
                // 邮件信息
                return MessengerChannelConstant.CHANNEL_EMAIL;
            }
            if (messageVO instanceof SmsMessageVO) {
                // 短信消息
                if (StringUtils.hasText(((SmsMessageVO) messageVO).getTemplateCode())) {
                    // 模板短信
                    if (StringUtils.hasText(messengerProperties.getSender().getAliyunSms().getAccessKeyId())) {
                        return MessengerChannelConstant.CHANNEL_SMS_ALIYUN;
                    }
                }
                return MessengerChannelConstant.CHANNEL_SMS;
            }
            if (messageVO instanceof SiteMessageVO) {
                // 站内信
                return MessengerChannelConstant.CHANNEL_SITE;
            }
            if (messageVO instanceof AppMessageVO) {
                // app消息
                if (StringUtils.hasText(messengerProperties.getSender().getJpush().getAppKey())) {
                    return MessengerChannelConstant.CHANNEL_APP_JPUSH;
                }
                return MessengerChannelConstant.CHANNEL_APP;
            }

            LOG.warn("获取消息渠道异常，不支持的消息类型：{}", messageVO.getClass().getName());
            return null;
        };
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        MessengerSenderSupport.setMessengerProperties(messengerProperties);

        var messagePropertiesChoose = applicationContext.getBeanProvider(MessagePropertiesChoose.class).getIfAvailable();
        if (messagePropertiesChoose != null) {
            MessengerSenderSupport.setMessagePropertiesChoose(messagePropertiesChoose);
        }

        var messageChannelChoose = applicationContext.getBeanProvider(MessageChannelChoose.class).getIfAvailable();
        if (messageChannelChoose != null) {
            MessengerSenderSupport.setMessageChannelChoose(messageChannelChoose);
        }

        var currentUserProvider = applicationContext.getBeanProvider(CurrentUserProvider.class).getIfAvailable();
        if (currentUserProvider != null) {
            MessengerSenderSupport.setCurrentUserProvider(currentUserProvider);
        }
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.applicationContext = applicationContext;
    }

    @Configuration
    @ConditionalOnClass(SecurityContext.class)
    static class SecurityConfig {

        @Bean
        @ConditionalOnMissingBean
        public CurrentUserProvider currentUserProviderDefault() {
            return () -> {
                var authentication = SecurityContextHolder.getContext().getAuthentication();

                return authentication == null ? null : authentication.getName();
            };
        }
    }
}
