package com.elitescloud.cloudt.messenger.config.support;

import cn.hutool.core.lang.Assert;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.util.RestTemplateFactory;
import com.elitescloud.boot.util.RestTemplateHelper;
import com.elitescloud.cloudt.common.base.RpcResult;
import com.elitescloud.cloudt.messenger.common.MessengerUriConstants;
import com.elitescloud.cloudt.messenger.common.MsgReceiveStatusEnum;
import com.elitescloud.cloudt.messenger.config.CloudtMessengerProperties;
import com.elitescloud.cloudt.messenger.config.MessengerSenderSupport;
import com.elitescloud.cloudt.messenger.message.AppMessageVO;
import com.elitescloud.cloudt.messenger.message.EmailMessageVO;
import com.elitescloud.cloudt.messenger.message.SiteMessageVO;
import com.elitescloud.cloudt.messenger.message.SmsMessageVO;
import com.elitescloud.cloudt.messenger.model.AbstractMessageVO;
import com.elitescloud.cloudt.messenger.model.dto.MsgResultDTO;
import com.elitescloud.coord.messenger.Application;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.client.RestTemplateBuilder;
import org.springframework.core.ParameterizedTypeReference;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpRequest;
import org.springframework.http.client.ClientHttpRequestExecution;
import org.springframework.http.client.ClientHttpRequestInterceptor;
import org.springframework.http.client.ClientHttpResponse;
import org.springframework.util.StringUtils;
import org.springframework.web.client.RestTemplate;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotEmpty;
import javax.validation.constraints.NotNull;
import java.io.IOException;
import java.util.*;
import java.util.function.Function;

/**
 * 消息发送提供者.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/22
 */
public class CloudtMessengerSender {
    private static final Logger LOG = LoggerFactory.getLogger(CloudtMessengerSender.class);
    private static CloudtMessengerSender messangerSender;

    private final RestTemplateHelper restTemplateHelper;

    private CloudtMessengerSender(CloudtMessengerProperties messengerProperties) {
        var restTemplate = this.buildRestTemplate(messengerProperties);
        this.restTemplateHelper = RestTemplateHelper.instance(restTemplate);
    }

    /**
     * 创建实例
     *
     * @return 发送实例
     */
    public static CloudtMessengerSender instance() {
        if (messangerSender == null) {
            return buildInstance();
        }
        return messangerSender;
    }

    /**
     * 发送消息
     *
     * @param messenger 消息对象
     * @param <T>       消息类型
     * @return 消息回执编码
     */
    public <T extends AbstractMessageVO> String send(T messenger) {
        var uri = this.detectSendUri(messenger);

        RpcResult<String> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.POST, new HttpEntity<>(messenger), new ParameterizedTypeReference<>() {
            });
        } catch (Exception e) {
            LOG.error("发送消息异常：", e);
            throw e;
        }

        LOG.info("发送消息{}结果：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        if (rpcResult.isSuccess()) {
            return rpcResult.getData();
        }
        return null;
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageId 消息ID
     * @return 消息状态
     */
    public String getMsgStatus(@NotBlank String messageId) {
        Assert.notBlank(messageId, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_STATUS + "?messageId={messageId}";
        RpcResult<String> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            }, messageId);
        } catch (Exception e) {
            LOG.error("查询消息状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return rpcResult.getData();
    }

    /**
     * 获取消息的发送状态
     *
     * @param messageIds 消息ID
     * @return ID与消息状态
     */
    public Map<String, String> queryMsgStatus(@NotEmpty Set<String> messageIds) {
        Assert.notEmpty(messageIds, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_STATUS_BATCH;
        RpcResult<Map<String, String>> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.POST, new HttpEntity<>(messageIds), new ParameterizedTypeReference<>() {
            });
        } catch (Exception e) {
            LOG.error("查询消息状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return ObjectUtil.defaultIfNull(rpcResult.getData(), Collections.emptyMap());
    }

    /**
     * 获取消息的接收状态
     *
     * @param messageId 消息ID
     * @return 账号与接收状态
     */
    public Map<String, String> queryMsgReceiveStatus(@NotBlank String messageId) {
        Assert.notBlank(messageId, "消息ID为空");
        var uri = MessengerUriConstants.MSG_RECEIVE_STATUS + "?messageId={messageId}";
        RpcResult<Map<String, String>> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            }, messageId);
        } catch (Exception e) {
            LOG.error("查询消息接收状态异常：", e);
            throw e;
        }

        LOG.info("查询消息{}接收状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return ObjectUtil.defaultIfNull(rpcResult.getData(), Collections.emptyMap());
    }

    /**
     * 查询消息发送结果
     *
     * @param messageId 消息ID
     * @return 发送结果
     */
    public MsgResultDTO querySendResult(@NotBlank String messageId) {
        Assert.notBlank(messageId, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_RESULT + "?messageId={messageId}";
        RpcResult<MsgResultDTO> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.GET, null, new ParameterizedTypeReference<>() {
            }, messageId);
        } catch (Exception e) {
            LOG.error("查询消息发送结果异常：", e);
            throw e;
        }

        LOG.info("查询消息{}发送结果：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return rpcResult.getData();
    }

    /**
     * 查询消息发送结果
     *
     * @param messageIds 消息ID
     * @return 发送结果
     */
    public List<MsgResultDTO> querySendResult(@NotEmpty Collection<String> messageIds) {
        Assert.notEmpty(messageIds, "消息ID为空");
        var uri = MessengerUriConstants.MSG_SEND_RESULT_BATCH;
        RpcResult<List<MsgResultDTO>> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.POST, new HttpEntity<>(messageIds, null), new ParameterizedTypeReference<>() {
            });
        } catch (Exception e) {
            LOG.error("查询消息发送结果异常：", e);
            throw e;
        }

        LOG.info("查询消息{}发送结果：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return rpcResult.getData();
    }

    /**
     * 更新消息的接收状态
     *
     * @param messageId 消息ID
     * @return 账号与接收状态
     */
    public String retryMsg(@NotBlank String messageId, Boolean sync) {
        Assert.notBlank(messageId, "消息ID为空");
        if (sync == null) {
            sync = false;
        }
        var uri = MessengerUriConstants.SENDER_RETRY + "?messageId={messageId}&sync={sync}";
        RpcResult<String> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.PUT, null, new ParameterizedTypeReference<>() {
            }, messageId, sync);
        } catch (Exception e) {
            LOG.error("重试发送消息异常：", e);
            throw e;
        }

        LOG.info("重试消息{}结果：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return rpcResult.getData();
    }

    /**
     * 更新消息的接收状态
     *
     * @param messageId 消息ID
     * @return 账号与接收状态
     */
    public Boolean updateMsgReceiveStatus(@NotBlank String messageId, @NotBlank String receiver,
                                          @NotNull MsgReceiveStatusEnum receiveStatus) {
        Assert.notBlank(messageId, "消息ID为空");
        Assert.notBlank(receiver, "接收人账号为空");
        Assert.notNull(receiveStatus, "接收状态为空");
        var uri = MessengerUriConstants.MSG_RECEIVE_STATUS_UPDATE + "?messageId={messageId}&receiver={receiver}&receiveStatus={receiveStatus}";
        RpcResult<Boolean> rpcResult = null;
        try {
            rpcResult = restTemplateHelper.exchange(uri, HttpMethod.PATCH, null, new ParameterizedTypeReference<>() {
            }, messageId, receiver, receiveStatus.name());
        } catch (Exception e) {
            LOG.error("更新消息接收状态异常：", e);
            throw e;
        }

        LOG.info("更新消息{}接收状态：{}, {}", uri, rpcResult.getMsg(), rpcResult.getData());
        return ObjectUtil.defaultIfNull(rpcResult.getData(), false);
    }

    private RestTemplate buildRestTemplate(CloudtMessengerProperties messengerProperties) {
        Function<RestTemplateBuilder, RestTemplateBuilder> restTemplateBuilderFunction = builder ->
                builder.rootUri(messengerProperties.getServerUrl())
                        .additionalInterceptors(new AuthorizationInterceptor(messengerProperties));
        return RestTemplateFactory.dynamicInstance(restTemplateBuilderFunction, Application.NAME);
    }

    private <T extends AbstractMessageVO> String detectSendUri(T messenger) {
        if (messenger instanceof SmsMessageVO) {
            return MessengerUriConstants.SENDER_SMS;
        }
        if (messenger instanceof EmailMessageVO) {
            return MessengerUriConstants.SENDER_EMAIL;
        }
        if (messenger instanceof SiteMessageVO) {
            return MessengerUriConstants.SENDER_SITE;
        }
        if (messenger instanceof AppMessageVO) {
            return MessengerUriConstants.SENDER_APP;
        }
        throw new IllegalStateException("发送失败，暂不支持的消息类型");
    }

    private static CloudtMessengerSender buildInstance() {
        Assert.state(MessengerSenderSupport.getMessengerProperties() != null, "消息发送初始化失败");
        synchronized (CloudtMessengerSender.class) {
            if (messangerSender != null) {
                return messangerSender;
            }

            // 初始化
            messangerSender = new CloudtMessengerSender(MessengerSenderSupport.getMessengerProperties());

            return messangerSender;
        }
    }

    static class AuthorizationInterceptor implements ClientHttpRequestInterceptor {

        private final CloudtMessengerProperties messengerProperties;

        public AuthorizationInterceptor(CloudtMessengerProperties messengerProperties) {
            this.messengerProperties = messengerProperties;
        }

        @Override
        public ClientHttpResponse intercept(HttpRequest request, byte[] body, ClientHttpRequestExecution execution) throws IOException {
            if (Boolean.TRUE.equals(messengerProperties.getAuthorization())) {
                var requestAttributes = RequestContextHolder.getRequestAttributes();
                if (requestAttributes != null) {
                    var originalRequest = ((ServletRequestAttributes) requestAttributes).getRequest();
                    var token = originalRequest.getHeader(HttpHeaders.AUTHORIZATION);
                    if (StringUtils.hasText(token)) {
                        request.getHeaders().add(HttpHeaders.AUTHORIZATION, token);
                    }
                }
            }
            return execution.execute(request, body);
        }
    }
}
