package com.elitescloud.cloudt.messenger.message.builder;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.cloudt.messenger.config.MessengerSenderSupport;
import com.elitescloud.cloudt.messenger.message.EmailMessageVO;
import com.elitescloud.cloudt.messenger.model.MessageAccountVO;
import com.elitescloud.cloudt.messenger.model.MessageAttachmentVO;
import io.swagger.annotations.ApiModelProperty;
import org.springframework.util.StringUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Email消息构建器.
 *
 * @author Kaiser（wang shao）
 * @date 2023/5/19
 */
public class EmailMessageBuilder extends AbstractMessageBuilder<EmailMessageBuilder, EmailMessageVO> {

    /**
     * 回复人
     */
    protected MessageAccountVO replyTo;

    /**
     * 抄送
     */
    protected List<MessageAccountVO> ccList;

    /**
     * 密送
     */
    protected List<MessageAccountVO> bccList;

    /**
     * 回执账号
     */
    protected String receiptAccount;

    /**
     * 附件
     */
    protected List<MessageAttachmentVO> attachments;

    @Override
    public EmailMessageVO build() {
        EmailMessageVO messageVO = new EmailMessageVO();
        messageVO.setReplyTo(this.replyTo);
        messageVO.setCcList(this.ccList);
        messageVO.setBccList(this.bccList);
        messageVO.setReceiptAccount(this.receiptAccount);
        messageVO.setAttachments(this.attachments);
        messageVO.setRichTextEmail(false);
        messageVO.setInlineAttachments(Collections.emptyMap());
        // 基本信息
        super.fillBaseMessage(messageVO);

        // 设置发件人，默认是账号信息
        messageVO.setSender(this.detectSender());

        return messageVO;
    }

    protected MessageAccountVO detectSender() {
        var emailProperties = MessengerSenderSupport.getMessengerProperties().getSender().getEmail();
        if (StringUtils.hasText(emailProperties.getUsername())) {
            return new MessageAccountVO(emailProperties.getUsername(), CharSequenceUtil.blankToDefault(emailProperties.getShowName(), emailProperties.getUsername()));
        }
        return null;
    }

    @Override
    public EmailMessageBuilder setRetryableOnFailure(Boolean retryableOnFailure) {
        return super.setRetryableOnFailure(retryableOnFailure);
    }

    @Override
    public EmailMessageBuilder setSubject(String subject) {
        return super.setSubject(subject);
    }

    @Override
    public EmailMessageBuilder setContent(String content) {
        return super.setContent(content);
    }

    @Override
    public EmailMessageBuilder setSender(MessageAccountVO sender) {
        return super.setSender(sender);
    }

    @Override
    public EmailMessageBuilder setReceiverList(List<MessageAccountVO> receiverList) {
        return super.setReceiverList(receiverList);
    }

    @Override
    public EmailMessageBuilder addReceiver(MessageAccountVO receiver) {
        return super.addReceiver(receiver);
    }

    public EmailMessageBuilder setReplyTo(MessageAccountVO replyTo) {
        this.replyTo = replyTo;
        return self();
    }

    public EmailMessageBuilder setCcList(List<MessageAccountVO> ccList) {
        this.ccList = ccList;
        return self();
    }

    public EmailMessageBuilder addCc(MessageAccountVO cc) {
        if (cc == null) {
            return self();
        }
        if (this.ccList == null) {
            this.ccList = new ArrayList<>();
        }
        this.ccList.add(cc);
        return self();
    }

    public EmailMessageBuilder addCc(List<MessageAccountVO> cc) {
        if (cc == null) {
            return self();
        }
        if (this.ccList == null) {
            this.ccList = new ArrayList<>();
        }
        this.ccList.addAll(cc);
        return self();
    }

    public EmailMessageBuilder setBccList(List<MessageAccountVO> bccList) {
        this.bccList = bccList;
        return self();
    }

    public EmailMessageBuilder addBcc(MessageAccountVO bcc) {
        if (bcc == null) {
            return self();
        }
        if (this.bccList == null) {
            this.bccList = new ArrayList<>();
        }
        this.bccList.add(bcc);
        return self();
    }

    public EmailMessageBuilder addBcc(List<MessageAccountVO> bcc) {
        if (bcc == null) {
            return self();
        }
        if (this.bccList == null) {
            this.bccList = new ArrayList<>();
        }
        this.bccList.addAll(bcc);
        return self();
    }

    public EmailMessageBuilder setReceiptAccount(String receiptAccount) {
        this.receiptAccount = receiptAccount;
        return self();
    }

    public EmailMessageBuilder setAttachments(List<MessageAttachmentVO> attachments) {
        this.attachments = attachments;
        return self();
    }

    public EmailMessageBuilder addAttachments(MessageAttachmentVO attachmentVO) {
        if (attachmentVO == null) {
            return self();
        }
        attachmentVO.validate();

        if (attachments == null) {
            attachments = new ArrayList<>(8);
        }
        attachments.add(attachmentVO);

        return self();
    }

    @Override
    protected EmailMessageBuilder self() {
        return this;
    }
}
