package com.elitescloud.boot.mq.tenant;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.common.constant.SpringCloudStreamConstant;
import com.elitescloud.boot.exception.BusinessException;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.support.ChannelInterceptor;

/**
 * 消息接收者拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/25
 */
@Log4j2
class TenantMessagingInputInterceptor implements ChannelInterceptor {

    @Override
    public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {
        Long tenantId = obtainTenantId(message);
        // 判断header是否有租户ID，如果有，则设置租户上下文
        if (tenantId != null) {
            SysTenantDTO tenant = TenantClient.getTenant(tenantId);
            if (tenant == null) {
                throw new BusinessException("MQ消息【" + obtainMessageId(message) + "】处理失败，未找到租户：" + tenantId);
            }
            TenantContextHolder.setCurrentTenant(tenant);
        }
        log.info("MQ消息【{}】消费...", tenantId);

        return message;
    }

    @Override
    public void afterSendCompletion(@NonNull Message<?> message, @NonNull MessageChannel channel, boolean sent, Exception ex) {
        Long tenantId = obtainTenantId(message);
        if (tenantId != null) {
            // 清除上下文
            TenantContextHolder.clearCurrentTenant();
        }

        if (ex == null) {
            log.info("MQ消息【{}】接收处理成功，所属租户{}", () -> obtainMessageId(message), () -> tenantId);
            return;
        }

        log.error("MQ消息【" + obtainMessageId(message) + "】接收处理失败，所属租户" + obtainMessageId(message), ex);
    }

    private String obtainMessageId(Message<?> message) {
        return ObjectUtil.defaultIfNull(message.getHeaders().getId(), "").toString();
    }

    private Long obtainTenantId(Message<?> message) {
        Object tenantId = message.getHeaders().get(SpringCloudStreamConstant.HEADER_TENANT_ID);
        if (tenantId == null) {
            return null;
        }

        return tenantId instanceof Long ? (Long) tenantId : Long.parseLong(tenantId.toString());
    }
}
