package com.elitescloud.boot.mq.config;

import cn.hutool.core.text.CharSequenceUtil;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.support.AbstractEnvironmentPostProcessor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.event.ApplicationContextInitializedEvent;
import org.springframework.context.ApplicationEvent;
import org.springframework.context.event.SmartApplicationListener;
import org.springframework.core.Ordered;
import org.springframework.core.env.ConfigurableEnvironment;
import org.springframework.core.env.Environment;
import org.springframework.core.env.StandardEnvironment;
import org.springframework.lang.NonNull;

import java.util.Map;

/**
 * MQ默认属性配置.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/20
 */
@Slf4j
class CloudtMqEnvironmentListener implements SmartApplicationListener, Ordered {

    @Override
    public boolean supportsEventType(Class<? extends ApplicationEvent> eventType) {
        return eventType.isAssignableFrom(ApplicationContextInitializedEvent.class);
    }

    @Override
    public void onApplicationEvent(@NonNull ApplicationEvent event) {
        if (event instanceof ApplicationContextInitializedEvent) {
            Environment env = ((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment();
            if (!(env instanceof StandardEnvironment)) {
                return;
            }

            StandardEnvironment environment = (StandardEnvironment) ((ApplicationContextInitializedEvent) event).getApplicationContext().getEnvironment();
            var propertySource = environment.getPropertySources().get(AbstractEnvironmentPostProcessor.PROPERTY_SOURCE_NAME);
            if (propertySource == null) {
                throw new IllegalStateException("缺少默认PropertySource");
            }
            Map<String, Object> defaultProperties = (Map<String, Object>) propertySource.getSource();

            createDefaultProperties(environment, defaultProperties);
        }
    }

    @Override
    public int getOrder() {
        return Ordered.HIGHEST_PRECEDENCE;
    }

    private Map<String, Object> createDefaultProperties(ConfigurableEnvironment environment, Map<String, Object> defaultProperties) {
        var topicPrefix = environment.getProperty(CloudtMqProperties.CONFIG_PREFIX + ".topic-prefix");
        var topic = CharSequenceUtil.isBlank(topicPrefix) ? MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL : topicPrefix + MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL;

        defaultProperties.put("spring.cloud.stream.bindings." + MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_OUTPUT + ".destination", topic);
        defaultProperties.put("spring.cloud.stream.bindings." + MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_INPUT + ".destination", topic);

        // 消费组
        var consumerGroup = environment.getProperty(CloudtMqProperties.CONFIG_PREFIX + ".consumer-group");
        if (CharSequenceUtil.isBlank(consumerGroup)) {
            consumerGroup = this.createConsumerGroup(environment);
        }

        log.info("CloudtMq: group is {}, topic is {}", consumerGroup, topic);
        defaultProperties.put("spring.cloud.stream.bindings." + MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_INPUT + ".group", consumerGroup);

        return defaultProperties;
    }

    private String createConsumerGroup(ConfigurableEnvironment environment) {
        String appName = environment.getProperty("spring.application.name");
        if (CharSequenceUtil.isBlank(appName)) {
            log.error("应用名称为空：spring.application.name");
            appName = "unknown";
        }
        return MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_INPUT_GROUP_PREFIX + appName;
    }
}
