package com.elitescloud.boot.mq.config.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import cn.hutool.crypto.digest.MD5;
import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.mq.common.BaseMessage;
import com.elitescloud.boot.mq.common.MessageDuplicateStrategy;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.mq.common.MessageQueueStorage;
import com.elitescloud.boot.mq.common.model.StoreMessageDTO;
import com.elitescloud.boot.mq.config.CloudtMqProperties;
import com.elitescloud.boot.redis.util.RedisUtils;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.Output;
import org.springframework.core.task.TaskExecutor;
import org.springframework.messaging.MessageChannel;

import java.io.Serializable;
import java.nio.charset.StandardCharsets;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.TimeUnit;

/**
 * 消息发送.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/14
 */
@Slf4j
public class DefaultMessageQueueTemplate implements MessageQueueTemplate {

    private final MessageCommonSupport messageCommonSupport;
    private final TaskExecutor taskExecutor;
    private final CloudtMqProperties mqProperties;
    private final RedisUtils redisUtils;
    private List<MessageQueueStorage> queueStorageList = Collections.emptyList();
    private List<MessageDuplicateStrategy> duplicateStrategyList = Collections.emptyList();
    private MessageChannel messageChannel;

    public DefaultMessageQueueTemplate(TaskExecutor taskExecutor, MessageCommonSupport messageCommonSupport,
                                       CloudtMqProperties mqProperties, RedisUtils redisUtils) {
        this.messageCommonSupport = messageCommonSupport;
        this.taskExecutor = taskExecutor;
        this.mqProperties = mqProperties;
        this.redisUtils = redisUtils;
    }

    @Override
    public <T extends Serializable> void sendMessage(String channel, T payload) {
        this.sendMessage(null, channel, payload);
    }

    @Override
    public <T extends Serializable> void sendMessage(String appCode, String channel, T payload) {
        Assert.notBlank(channel, "消息渠道为空");
        Assert.notNull(payload, "消息体为空");
        var startTime = LocalDateTime.now();
        var messageId = messageCommonSupport.generateMessageId();
        if (!this.canSend(appCode)) {
            log.info("{}的消息发送已禁用", appCode);
            // 持久化消息
            this.storageRecord(appCode, channel, messageId, payload, startTime, true, false, false, "已禁用该应用发送");
            return;
        }

        var listeners = messageCommonSupport.getListenerMap().getOrDefault(channel, Collections.emptyList());
        if (listeners.isEmpty()) {
            // 未找到本地消费者，则走MQ
            this.executeSend(() -> messageCommonSupport.publishMqMessage(messageChannel, channel, messageId, payload), appCode, channel,
                    messageId, payload, false, startTime);
            return;
        }

        log.info("本地消息：{}, 消费者：{}", channel, listeners.size());
        // 执行发送
        this.executeSend(() -> taskExecutor.execute(() -> {
            for (var listener : listeners) {
                try {
                    listener.getMessageQueueListener().onConsume(channel, payload);
                } catch (Throwable e) {
                    log.error("{}消费消息异常：", listener.getClass().getName(), e);
                }
            }
        }), appCode, channel, messageId, payload, true, startTime);
    }

    @Override
    public <T extends Serializable> void publishMessage(String channel, T payload) {
        this.publishMessage(null, channel, payload);
    }

    @Override
    public <T extends Serializable> void publishMessage(String appCode, String channel, T payload) {
        Assert.notBlank(channel, "消息渠道为空");
        Assert.notNull(payload, "消息体为空");
        LocalDateTime startTime = LocalDateTime.now();
        var messageId = messageCommonSupport.generateMessageId();
        if (!this.canSend(appCode)) {
            log.info("{}的消息发送已禁用", appCode);
            // 持久化消息
            this.storageRecord(appCode, channel, messageId, payload, startTime, false, false, false, "已禁用该应用发送");
            return;
        }

        // 执行发送
        this.executeSend(() -> messageCommonSupport.publishMqMessage(messageChannel, channel, messageId, payload), appCode, channel, messageId, payload, false, startTime);
    }

    @Autowired
    @Output(MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_OUTPUT)
    public void setMessageChannel(MessageChannel messageChannel) {
        this.messageChannel = messageChannel;
    }

    public void setQueueStorageList(List<MessageQueueStorage> queueStorageList) {
        this.queueStorageList = ObjectUtil.defaultIfNull(queueStorageList, Collections.emptyList());
    }

    public void setDuplicateStrategyList(List<MessageDuplicateStrategy> duplicateStrategyList) {
        this.duplicateStrategyList = ObjectUtil.defaultIfNull(duplicateStrategyList, Collections.emptyList());
    }

    private boolean canSend(String appCode) {
        if (CharSequenceUtil.isBlank(appCode)) {
            // 默认均可发送
            return true;
        }
        var apps = mqProperties.getDisabledApps();
        if (CollUtil.isEmpty(apps) || apps.contains("*")) {
            return true;
        }

        return !apps.contains(appCode);
    }

    private void executeSend(Runnable sendThread, String appCode, String channel, String messageId, Object payload, boolean local, LocalDateTime startTime) {
        // 判断消息是否重复
        if (this.isDuplicate(appCode, channel, payload)) {
            this.storageRecord(appCode, channel, messageId, payload, startTime, local, false, false, "消息已重复");
            return;
        }

        // 保存发送记录
        this.storageRecord(appCode, channel, messageId, payload, startTime, local, local, true, null);

        Throwable exp = null;
        try {
            sendThread.run();
        } catch (Throwable e) {
            exp = e;
            throw e;
        } finally {
            // 更新发送结果
            this.updateSendResult(messageId, local, exp);
        }
    }

    private boolean isDuplicate(String appCode, String channel, Object payload) {
        if (Boolean.TRUE.equals(mqProperties.getAllowDuplicate())) {
            // 允许重复
            return false;
        }

        // 自定义判重策略
        if (!duplicateStrategyList.isEmpty()) {
            for (MessageDuplicateStrategy messageDuplicateStrategy : duplicateStrategyList) {
                if (messageDuplicateStrategy.isDuplicate(appCode, channel, payload)) {
                    log.info("自定义消息判重策略判断已重复：{}，{}", appCode, channel);
                    return true;
                }
            }
            return false;
        }

        // 默认判重策略
        String digest = null;
        try {
            String txt = appCode + "&" + channel + "&" + messageCommonSupport.messageToString(payload);
            digest = MD5.create().digestHex(txt, StandardCharsets.UTF_8);
        } catch (Exception e) {
            throw new IllegalStateException("判断消息是否重复异常", e);
        }

        var obj = redisUtils.get(digest);
        if (obj != null) {
            // 已重复
            log.info("消息已重复：{}，{}", appCode, channel);
            return true;
        }
        redisUtils.set(digest, "true", mqProperties.getDuplicateInterval().toSeconds(), TimeUnit.SECONDS);
        return false;
    }

    private String storageRecord(String appCode, String channel, String messageId, Object payload, LocalDateTime sendTime, boolean local, boolean success, boolean retry, String failReason) {
        if (Boolean.FALSE.equals(mqProperties.getAllowStorage())) {
            // 不存储
            return null;
        }
        if (queueStorageList.isEmpty()) {
            // 不需要持久化
            return null;
        }

        LocalDateTime endTime = LocalDateTime.now();
        StoreMessageDTO messageDTO = new StoreMessageDTO();
        messageDTO.setMessageId(messageId);
        messageDTO.setAppCode(appCode);
        messageDTO.setChannel(channel);
        messageDTO.setMessageContent(messageCommonSupport.messageToString(payload));
        if (payload instanceof BaseMessage) {
            messageDTO.setBusinessKey(((BaseMessage) payload).getBusinessKey());
        }
        messageDTO.setSendTime(sendTime);
        messageDTO.setFinishTime(endTime);
        messageDTO.setLocal(local);
        // 本地消费的话，直接就是发送成功
        messageDTO.setSuccess(success);
        messageDTO.setFailReason(failReason);
        messageDTO.setRetryable(retry);

        for (MessageQueueStorage messageQueueStorage : queueStorageList) {
            try {
                messageQueueStorage.saveMessage(messageDTO);
            } catch (Throwable e) {
                log.error("持久化消息发送记录异常：", e);
            }
        }

        return messageId;
    }

    public void updateSendResult(String messageId, boolean local, Throwable e) {
        if (CharSequenceUtil.isBlank(messageId)) {
            return;
        }

        String failReason = e == null ? null : e.getMessage();
        CompletableFuture.runAsync(() -> {
                    if (local) {
                        // 本地调用，则直接更新消费结果
                        for (MessageQueueStorage messageQueueStorage : queueStorageList) {
                            try {
                                messageQueueStorage.updateConsumeResult(messageId, e == null, failReason);
                            } catch (Throwable t) {
                                log.error("更新消费结果异常：", e);
                            }
                        }
                        return;
                    }

                    for (MessageQueueStorage messageQueueStorage : queueStorageList) {
                        try {
                            messageQueueStorage.updateSendResult(messageId, e == null, failReason);
                        } catch (Throwable t) {
                            log.error("更新发送结果异常：", e);
                        }
                    }
                }, taskExecutor)
                .whenComplete((res, exp) -> {
                    if (exp != null) {
                        log.error("更新消息结果异常：{}", messageId, exp);
                    }
                });
    }
}
