package com.elitescloud.boot.mq.config.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.lang.Assert;
import cn.hutool.core.lang.UUID;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.RandomUtil;
import com.elitescloud.boot.SpringContextHolder;
import com.elitescloud.boot.mq.MessageQueueListener;
import com.elitescloud.boot.mq.common.BaseMessage;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.util.JSONUtil;
import lombok.extern.slf4j.Slf4j;
import org.springframework.integration.support.MutableMessageBuilder;
import org.springframework.messaging.MessageChannel;

import javax.validation.constraints.NotBlank;
import javax.validation.constraints.NotNull;
import java.io.Serializable;
import java.lang.reflect.ParameterizedType;
import java.util.*;
import java.util.stream.Collectors;

/**
 * 消息服务支撑.
 *
 * @author Kaiser（wang shao）
 * @date 2023/9/14
 */
@Slf4j
public class MessageCommonSupport {

    private Map<String, List<MessageCommonSupport.MessageQueueListenerWrapper>> listenerMap;

    /**
     * 获取监听者
     *
     * @return
     */
    Map<String, List<MessageCommonSupport.MessageQueueListenerWrapper>> getListenerMap() {
        if (listenerMap == null) {
            listenerMap = this.initMessageQueueListener(SpringContextHolder.getObjectProvider(MessageQueueListener.class).stream().collect(Collectors.toList()));
        }
        return listenerMap;
    }

    /**
     * 发布MQ消息
     *
     * @param messageChannel 消息渠道
     * @param channel        业务渠道
     * @param payload        负载数据
     */
    <T extends Serializable> void publishMqMessage(@NotNull MessageChannel messageChannel, @NotBlank String channel,
                                                   @NotBlank String messageId, @NotNull T payload) {
        Assert.notBlank(messageId, "消息ID为空");
        Assert.notNull(payload, "发布消息为空");
        log.info("发布消息：{}, {}", channel, messageId);

        // 转换消息体
        String payloadStr = null;
        if (payload instanceof String) {
            payloadStr = (String) payload;
        } else {
            payloadStr = JSONUtil.toJsonString(payload, true, () -> channel + "序列化消息体异常");
        }

        var messageBuilder = MutableMessageBuilder.withPayload(payloadStr)
                .setHeader(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_CHANNEL_ORIGINAL, channel)
                .setHeader(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_ID, messageId)
                .setHeader(MessageQueueConstant.HEADER_KEYS, buildMessageKeys(payload, messageId));

        var message = messageBuilder.build();

        messageChannel.send(message);
    }

    <T extends Serializable> String buildMessageKeys(T payload, String messageId) {
        List<String> keys = new ArrayList<>(4);
        keys.add(messageId);
        if (payload instanceof BaseMessage) {
            String businessKey = ((BaseMessage) payload).getBusinessKey();
            if (CharSequenceUtil.isNotBlank(businessKey)) {
                keys.add(businessKey);
            }
        } else if (payload instanceof Map) {
            Object businessKey = ((Map<?, ?>) payload).get("businessKey");
            if (businessKey instanceof String && CharSequenceUtil.isNotBlank((String) businessKey)) {
                keys.add((String) businessKey);
            }
        }

        return String.join(" ", keys);
    }

    /**
     * 创建消息ID
     *
     * @return 消息ID
     */
    String generateMessageId() {
        return UUID.fastUUID().toString(true) + ":" + RandomUtil.randomString(6);
    }

    /**
     * 消息转json字符串
     *
     * @param obj
     * @return
     */
    String messageToString(Object obj) {
        if (obj == null) {
            return null;
        }

        return JSONUtil.toJsonString(obj, true, () -> "消息序列化异常");
    }

    /**
     * 消息反序列化
     *
     * @param message
     * @param clazz
     * @param <T>
     * @return
     */
    <T> T readMessage(String message, Class<T> clazz) {
        return JSONUtil.json2Obj(message, clazz, true, () -> "消息反序列化异常");
    }

    private Map<String, List<MessageCommonSupport.MessageQueueListenerWrapper>> initMessageQueueListener(List<MessageQueueListener> listenerList) {
        if (CollUtil.isEmpty(listenerList)) {
            return Collections.emptyMap();
        }

        Map<String, List<MessageCommonSupport.MessageQueueListenerWrapper>> initListenerMap = new HashMap<>();
        for (MessageQueueListener<?> listener : listenerList) {
            Assert.notEmpty(listener.channels(), listener.getClass().getName() + "中渠道未配置");
            for (String channel : listener.channels()) {
                initListenerMap.computeIfAbsent(channel, c -> new ArrayList<>(8))
                        .add(new MessageCommonSupport.MessageQueueListenerWrapper(listener));
            }
        }
        return initListenerMap;
    }

    static class MessageQueueListenerWrapper {
        private final MessageQueueListener<?> messageQueueListener;
        private Class<?> messageType;

        public MessageQueueListenerWrapper(MessageQueueListener<?> messageQueueListener) {
            this.messageQueueListener = messageQueueListener;
            this.init();
        }

        public MessageQueueListener getMessageQueueListener() {
            return messageQueueListener;
        }

        public Class<?> getMessageType() {
            return messageType;
        }

        private void init() {
            this.messageType = this.obtainMessageType();
        }

        private Class<?> obtainMessageType() {
            var genericInterfaces = messageQueueListener.getClass().getGenericInterfaces();
            if (genericInterfaces.length > 0 && genericInterfaces[0] instanceof ParameterizedType) {
                var typeArguments = ((ParameterizedType) genericInterfaces[0]).getActualTypeArguments();
                if (typeArguments.length > 0) {
                    var temp = typeArguments[0];
                    if (temp instanceof Class) {
                        return (Class<?>) temp;
                    }
                }
            }
            throw new IllegalStateException(messageQueueListener.getClass().getName() + "的消息类型为空或不支持");
        }
    }
}
