package com.elitescloud.boot.mq.config.support;

import cn.hutool.core.collection.CollUtil;
import cn.hutool.core.exceptions.ExceptionUtil;
import cn.hutool.core.text.CharSequenceUtil;
import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.mq.common.MessageQueueStorage;
import com.elitescloud.boot.util.ExceptionsUtil;
import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.stream.annotation.StreamListener;
import org.springframework.core.task.TaskExecutor;
import org.springframework.messaging.Message;

import java.io.Serializable;
import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.locks.Lock;

/**
 * 消息消费代理.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/14
 */
@Slf4j
public class MessageQueueConsumerDelegate {

    private final MessageCommonSupport messageCommonSupport;
    private final TaskExecutor taskExecutor;
    private List<MessageQueueStorage> queueStorageList = Collections.emptyList();

    public MessageQueueConsumerDelegate(MessageCommonSupport messageCommonSupport, TaskExecutor taskExecutor) {
        this.messageCommonSupport = messageCommonSupport;
        this.taskExecutor = taskExecutor;
    }

    @SuppressWarnings("unchecked")
    @StreamListener(MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_INPUT)
    public void consumer(Message<String> message) {
        String originalChannel = message.getHeaders().get(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_CHANNEL_ORIGINAL, String.class);
        if (CharSequenceUtil.isBlank(originalChannel)) {
            log.info("未找到消息的channel，忽略消息");
            return;
        }
        if (CharSequenceUtil.isBlank(message.getPayload())) {
            log.info("消息体为空，忽略消息");
            return;
        }

        // 回调消费者
        var listeners = messageCommonSupport.getListenerMap().getOrDefault(originalChannel, Collections.emptyList());
        if (listeners.isEmpty()) {
            return;
        }
        String messageId = message.getHeaders().get(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_ID, String.class);
        Throwable exp = null;
        var start = System.currentTimeMillis();
        for (MessageCommonSupport.MessageQueueListenerWrapper listener : listeners) {
            log.info("{}开始消费消息：{}", listener.getClass().getName(), messageId);
            try {
                var msg = messageCommonSupport.readMessage(message.getPayload(), listener.getMessageType());
                listener.getMessageQueueListener().onConsume(originalChannel, (Serializable) msg);
                exp = null;
            } catch (Throwable e) {
                log.error("消费者{}消费消息{}异常：", listener.getMessageQueueListener().getClass().getName(), message.getPayload(), e);
                exp = e;
            }
        }
        var costTime = System.currentTimeMillis() - start;

        // 更新消费结果
        this.updateConsumeResult(messageId, costTime, exp);
    }

    public void setQueueStorageList(List<MessageQueueStorage> queueStorageList) {
        this.queueStorageList = ObjectUtil.defaultIfNull(queueStorageList, Collections.emptyList());
    }

    private void updateConsumeResult(String messageId, long costTime, Throwable exp) {
        if (CollUtil.isEmpty(queueStorageList)) {
            return;
        }
        if (CharSequenceUtil.isBlank(messageId)) {
            log.warn("消息ID为空，忽略更新消费结果");
            return;
        }

        // 没有异常就标识消费成功
        CompletableFuture.runAsync(() -> {
                    var success = exp == null;
                    var expMsg = exp == null ? null : ExceptionsUtil.stackTraceAllToString(exp);

                    for (MessageQueueStorage storage : queueStorageList) {
                        storage.updateConsumeResult(messageId, success, expMsg, costTime);
                    }
                }, taskExecutor)
                .whenComplete((res, e) -> {
                    if (e != null) {
                        log.error("更新消费结果异常：{}", messageId, e);
                    }
                });

    }
}
