package com.elitescloud.boot.mq.config.support;

import cn.hutool.core.util.ObjUtil;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.mq.common.MessageRetryProvider;
import com.elitescloud.boot.mq.common.MessageRetryService;
import com.elitescloud.boot.mq.common.model.RetryMessageDTO;
import com.elitescloud.boot.mq.config.CloudtMqProperties;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.task.retry.AbstractRetryService;
import com.elitescloud.boot.util.JSONUtil;
import lombok.extern.slf4j.Slf4j;
import org.redisson.api.RedissonClient;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cloud.stream.annotation.Output;
import org.springframework.messaging.MessageChannel;

import java.io.Serializable;
import java.time.Duration;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 消息重试服务.
 *
 * @author Kaiser（wang shao）
 * @date 2023/8/28
 */
@Slf4j
public class MessageRetryServiceImpl extends AbstractRetryService<RetryMessageDTO> implements MessageRetryService {
    private final MessageCommonSupport messageCommonSupport;
    private final CloudtMqProperties mqProperties;
    private MessageChannel messageChannel;

    public MessageRetryServiceImpl(MessageCommonSupport messageCommonSupport,
                                   RedissonClient redissonClient, MessageRetryProvider messageRetryProvider,
                                   CloudtMqProperties mqProperties, TenantDataIsolateProvider tenantDataIsolateProvider) {
        super(messageRetryProvider, redissonClient, tenantDataIsolateProvider);
        this.messageCommonSupport = messageCommonSupport;
        this.mqProperties = mqProperties;
    }

    @Autowired
    @Output(MessageQueueConstant.CLOUDT_MESSAGE_CHANNEL_OUTPUT)
    public void setMessageChannel(MessageChannel messageChannel) {
        this.messageChannel = messageChannel;
    }

    @Override
    public String generateMessageId() {
        return messageCommonSupport.generateMessageId();
    }

    @Override
    protected boolean supportRetry() {
        return Boolean.TRUE.equals(mqProperties.getSupportRetry());
    }

    @Override
    protected int retryTimes() {
        return ObjUtil.defaultIfNull(mqProperties.getRetryTimes(), 0);
    }

    @Override
    protected List<Duration> retryIntervals() {
        return mqProperties.getRetryIntervals();
    }

    @Override
    protected void executeTask(RetryMessageDTO task) {
        tenantDataIsolateProvider.byTenantDirectly(() -> {
            HashMap<String, ? extends Serializable> message = JSONUtil.json2Obj(task.getMessageContent());
            messageCommonSupport.publishMqMessage(messageChannel, task.getChannel(), task.getTaskId(),
                    message == null ? task.getMessageContent() : message);
            return null;
        }, task.getSysTenantId());
    }
}
