package com.elitescloud.boot.mq.tenant;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.constant.SpringCloudStreamConstant;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.support.ChannelInterceptor;
import org.springframework.util.StringUtils;

/**
 * 消息接收者拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/25
 */
@Log4j2
class TenantMessagingInputInterceptor implements ChannelInterceptor {

    @Override
    public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {
        Long tenantId = obtainTenantId(message);
        // 判断header是否有租户ID，如果有，则设置租户上下文
        if (tenantId != null) {
            SysTenantDTO tenant = TenantClient.getTenant(tenantId);
            if (tenant == null) {
                log.error("MQ消息【{}】处理失败，未找到租户：{}", obtainMessageId(message), tenantId);
                return null;
            }
            TenantContextHolder.setCurrentTenant(tenant);
        }
        log.debug("MQ消息【{}】消费...", tenantId);

        return message;
    }

    @Override
    public void afterSendCompletion(@NonNull Message<?> message, @NonNull MessageChannel channel, boolean sent, Exception ex) {
        Long tenantId = obtainTenantId(message);
        if (tenantId != null) {
            // 清除上下文
            TenantContextHolder.clearCurrentTenant();
        }

        if (ex == null) {
            log.debug("MQ消息【{}】接收处理成功，所属租户{}", () -> obtainMessageId(message), () -> tenantId);
            return;
        }

        log.error("MQ消息【" + obtainMessageId(message) + "】接收处理失败，所属租户" + tenantId, ex);
    }

    private String obtainMessageId(Message<?> message) {
        var messageId = message.getHeaders().get(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_ID);
        if (messageId != null) {
            return messageId.toString();
        }
        return ObjectUtil.defaultIfNull(message.getHeaders().getId(), "").toString();
    }

    private Long obtainTenantId(Message<?> message) {
        Object tenantId = message.getHeaders().get(SpringCloudStreamConstant.HEADER_TENANT_ID);
        if (tenantId == null) {
            return null;
        }

        return tenantId instanceof Long ? (Long) tenantId : Long.parseLong(tenantId.toString());
    }
}
