package com.elitescloud.boot.mq.config;

import com.elitescloud.boot.constant.SpringCloudStreamConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.boot.tenant.client.common.TenantClient;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.support.ChannelInterceptor;
import org.springframework.messaging.support.MessageBuilder;
import org.springframework.messaging.support.MessageHeaderAccessor;

/**
 * 消息发送拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/25
 */
@Log4j2
class CloudtMessagingOutputInterceptor implements ChannelInterceptor {

    private final TenantClientProvider tenantClientProvider;

    public CloudtMessagingOutputInterceptor(TenantClientProvider tenantClientProvider) {
        this.tenantClientProvider = tenantClientProvider;
    }

    @Override
    public Message<?> preSend(Message<?> message, MessageChannel channel) {
        var tenantId = TenantConstant.DEFAULT_TENANT_ID;
        if (tenantClientProvider.enabledTenant()) {
            // 启用租户
            SysTenantDTO tenant = TenantClient.getSessionTenant();
            if (tenant == null) {
                return message;
            }
            tenantId = tenant.getId();
        }

        // 判断当前租户，如果租户存在，则传递租户header
        MessageHeaderAccessor headerAccessor = MessageHeaderAccessor.getMutableAccessor(message);
        headerAccessor.setHeaderIfAbsent(SpringCloudStreamConstant.HEADER_TENANT_ID, tenantId.toString());

        // 返回消息
        return MessageBuilder.withPayload(message.getPayload())
                .copyHeaders(headerAccessor.getMessageHeaders())
                .build();

    }

    @Override
    public void afterSendCompletion(Message<?> message, MessageChannel channel, boolean sent, Exception ex) {
        if (ex == null) {
            log.info("MQ消息【{}】发送成功，所属租户{}！", () -> message.getHeaders().getId(),
                    () -> message.getHeaders().get(SpringCloudStreamConstant.HEADER_TENANT_ID));
            return;
        }

        log.error("MQ消息发送失败：", ex);
    }
}
