package com.elitescloud.boot.mq.config;

import com.elitescloud.boot.mq.MessageQueueTemplate;
import com.elitescloud.boot.mq.common.MessageDuplicateStrategy;
import com.elitescloud.boot.mq.common.MessageQueueStorage;
import com.elitescloud.boot.mq.common.MessageRetryProvider;
import com.elitescloud.boot.mq.common.MessageRetryService;
import com.elitescloud.boot.mq.config.support.*;
import com.elitescloud.boot.provider.TenantDataIsolateProvider;
import com.elitescloud.boot.redis.util.RedisUtils;
import org.redisson.api.RedissonClient;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.stream.annotation.EnableBinding;
import org.springframework.context.annotation.Bean;
import org.springframework.core.task.TaskExecutor;

import java.util.List;

/**
 * Cloudt MQ自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 3/13/2023
 */
@EnableBinding(CloudtMessageChannel.class)
@ConditionalOnProperty(prefix = CloudtMqProperties.CONFIG_PREFIX, name = "enabled", havingValue = "true", matchIfMissing = true)
@EnableConfigurationProperties(CloudtMqProperties.class)
public class CloudtMqAutoConfiguration {

    private final CloudtMqProperties mqProperties;

    public CloudtMqAutoConfiguration(CloudtMqProperties mqProperties) {
        this.mqProperties = mqProperties;
    }

    @Bean
    public MessageCommonSupport messageCommonSupport() {
        return new MessageCommonSupport();
    }

    @Bean
    public MessageQueueTemplate messageQueueTemplateDefault(TaskExecutor taskExecutor,
                                                            MessageCommonSupport messageCommonSupport,
                                                            RedisUtils redisUtils,
                                                            List<MessageQueueStorage> queueStorageList,
                                                            List<MessageDuplicateStrategy> duplicateStrategyList) {
        DefaultMessageQueueTemplate template = new DefaultMessageQueueTemplate(taskExecutor, messageCommonSupport, mqProperties, redisUtils);
        template.setQueueStorageList(queueStorageList);
        template.setDuplicateStrategyList(duplicateStrategyList);
        return template;
    }

    @Bean
    @ConditionalOnProperty(prefix = CloudtMqProperties.CONFIG_PREFIX, name = "allow-consume", havingValue = "true", matchIfMissing = true)
    public MessageQueueConsumerDelegate messageQueueConsumerDelegate(TaskExecutor taskExecutor,
                                                                     MessageCommonSupport messageCommonSupport,
                                                                     List<MessageQueueStorage> queueStorageList) {
        var consumerDelegate = new MessageQueueConsumerDelegate(messageCommonSupport, taskExecutor);
        consumerDelegate.setQueueStorageList(queueStorageList);
        return consumerDelegate;
    }

    @Bean
    @ConditionalOnBean(MessageRetryProvider.class)
    @ConditionalOnProperty(prefix = CloudtMqProperties.CONFIG_PREFIX, name = "support-retry", havingValue = "true")
    public MessageRetryService messageRetryService(MessageCommonSupport messageCommonSupport,
                                                   RedissonClient redissonClient,
                                                   MessageRetryProvider messageRetryProvider,
                                                   TenantDataIsolateProvider tenantDataIsolateProvider) {
        return new MessageRetryServiceImpl(messageCommonSupport, redissonClient, messageRetryProvider, mqProperties, tenantDataIsolateProvider);
    }
}
