package com.elitescloud.boot.mq.config;

import cn.hutool.core.util.ObjectUtil;
import com.elitescloud.boot.base.AbstractLogTraceHandler;
import com.elitescloud.boot.constant.SpringCloudStreamConstant;
import com.elitescloud.boot.constant.TenantConstant;
import com.elitescloud.boot.context.ExecutorContextHolder;
import com.elitescloud.boot.context.TenantContextHolder;
import com.elitescloud.boot.mq.common.MessageQueueConstant;
import com.elitescloud.boot.provider.TenantClientProvider;
import com.elitescloud.cloudt.system.dto.SysTenantDTO;
import lombok.extern.log4j.Log4j2;
import org.springframework.lang.NonNull;
import org.springframework.messaging.Message;
import org.springframework.messaging.MessageChannel;
import org.springframework.messaging.support.ChannelInterceptor;

/**
 * 消息接收者拦截器.
 *
 * @author Kaiser（wang shao）
 * @date 2022/5/25
 */
@Log4j2
class CloudtMessagingInputInterceptor extends AbstractLogTraceHandler implements ChannelInterceptor {

    private final TenantClientProvider tenantClientProvider;

    public CloudtMessagingInputInterceptor(String applicationName, TenantClientProvider tenantClientProvider) {
        super(applicationName);
        this.tenantClientProvider = tenantClientProvider;
    }

    @Override
    public Message<?> preSend(@NonNull Message<?> message, @NonNull MessageChannel channel) {
        // 日志
        super.putTraceId();

        Long tenantId = obtainTenantId(message);
        // 判断header是否有租户ID，如果有，则设置租户上下文
        if (tenantId != null) {
            SysTenantDTO tenant = tenantClientProvider.getTenant(tenantId);
            if (tenant == null) {
                log.error("MQ消息【{}】处理失败，未找到租户：{}", obtainMessageId(message), tenantId);
                return null;
            }
            TenantContextHolder.setCurrentTenant(tenant);
        }
        ExecutorContextHolder.create(ExecutorContextHolder.Source.MQ, null, false);
        log.debug("MQ消息【{}】消费...", tenantId);

        return message;
    }

    @Override
    public void afterSendCompletion(@NonNull Message<?> message, @NonNull MessageChannel channel, boolean sent, Exception ex) {
        // 清楚日志
        super.clearTraceId();

        Long tenantId = obtainTenantId(message);
        if (tenantId != null) {
            // 清除上下文
            TenantContextHolder.clearCurrentTenant();
        }
        ExecutorContextHolder.clear();

        if (ex == null) {
            log.debug("MQ消息【{}】接收处理成功，所属租户{}", () -> obtainMessageId(message), () -> tenantId);
            return;
        }

        log.error("MQ消息【" + obtainMessageId(message) + "】接收处理失败，所属租户" + tenantId, ex);
    }

    private String obtainMessageId(Message<?> message) {
        var messageId = message.getHeaders().get(MessageQueueConstant.HEADER_CLOUDT_MESSAGE_ID);
        if (messageId != null) {
            return messageId.toString();
        }
        return ObjectUtil.defaultIfNull(message.getHeaders().getId(), "").toString();
    }

    private Long obtainTenantId(Message<?> message) {
        if (!tenantClientProvider.enabledTenant()) {
            // 未启用租户
            return null;
        }

        // 从请求头取租户ID
        Object tenantId = message.getHeaders().get(SpringCloudStreamConstant.HEADER_TENANT_ID);
        if (tenantId == null || TenantConstant.DEFAULT_TENANT_ID.toString().equals(tenantId.toString())) {
            return null;
        }

        try {
            return tenantId instanceof Long ? (Long) tenantId : Long.parseLong(tenantId.toString());
        } catch (NumberFormatException e) {
            log.error("租户ID：{}", tenantId, e);
            throw new IllegalStateException("租户ID解析异常：", e);
        }
    }
}
