package com.elitescloud.boot.openfeign.config;

import com.elitescloud.boot.constant.ClassNameConstant;
import feign.Response;
import feign.codec.ErrorDecoder;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcRegistrations;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.client.loadbalancer.LoadBalancerClient;
import org.springframework.cloud.loadbalancer.config.BlockingLoadBalancerClientAutoConfiguration;
import org.springframework.cloud.loadbalancer.support.LoadBalancerClientFactory;
import org.springframework.cloud.openfeign.EnableFeignClients;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.lang.NonNull;
import org.springframework.web.servlet.mvc.method.annotation.RequestMappingHandlerMapping;

/**
 * OpenFeign自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/9/21
 */
@Configuration
@EnableConfigurationProperties(CloudtOpenFeignProperties.class)
@EnableFeignClients(basePackages = {"com.elitescloud", "com.elitesland"})
@AutoConfigureBefore({BlockingLoadBalancerClientAutoConfiguration.class})
@Log4j2
class CloudtOpenFeignAutoConfiguration {

    @Value("${spring.application.name:#{null}}")
    private String applicationName;
    @Value("${server.port:8080}")
    private Integer port;

    private final CloudtOpenFeignProperties properties;

    public CloudtOpenFeignAutoConfiguration(CloudtOpenFeignProperties properties) {
        this.properties = properties;
    }

    /**
     * 自定义负载均衡
     *
     * @param loadBalancerClientFactory
     * @return
     */
    @Bean
    public LoadBalancerClient blockingLoadBalancerClient(LoadBalancerClientFactory loadBalancerClientFactory) {
        return new CloudtBlockingLoadBalancerClient(applicationName, port, loadBalancerClientFactory);
    }

    /**
     * 处理webmvc的handler
     * <p>
     * 避免FeignClient的重复注册
     *
     * @return
     */
    @Bean
    public WebMvcRegistrations feignWebRegistrations() {
        return new WebMvcRegistrations() {
            @Override
            public RequestMappingHandlerMapping getRequestMappingHandlerMapping() {
                return new RequestMappingHandlerMapping() {
                    @Override
                    protected boolean isHandler(@NonNull Class<?> beanType) {
                        return super.isHandler(beanType) && !beanType.isAnnotationPresent(FeignClient.class);
                    }
                };
            }
        };
    }

    /**
     * 用户上下文信息传递interceptor
     *
     * @return
     */
    @Bean
    @ConditionalOnClass(name = ClassNameConstant.AUTHORIZATION_CLIENT)
    public FeignAuthenticationContextInterceptor feignAuthenticationContextInterceptor() {
        return new FeignAuthenticationContextInterceptor();
    }

    @Bean
    public CloudtHeaderInterceptor cloudtHeaderInterceptor() {
        return new CloudtHeaderInterceptor();
    }

    /**
     * traceId传递
     *
     * @return
     */
    @Bean
    public FeignTraceLogInterceptor feignLogTraceHandler() {
        return new FeignTraceLogInterceptor(applicationName);
    }

    /**
     * 异常打印
     *
     * @return
     */
    @Bean
    public ErrorDecoder errorDecoder() {
        return new ErrorDecoder.Default() {
            @Override
            public Exception decode(String methodKey, Response response) {
                var exp = super.decode(methodKey, response);
                log.error("OpenFeign处理异常：", exp);

                return exp;
            }
        };
    }

    /**
     * bean后置处理
     *
     * @return
     */
    @Bean
    @ConditionalOnProperty(prefix = CloudtOpenFeignProperties.CONFIG_PREFIX, name = "direct-local", havingValue = "true", matchIfMissing = true)
    public BeanPostProcessor openFeignBeanPostProcessor() {
        return new OpenFeignBeanPostProcessor(applicationName, properties);
    }
}
