package com.elitescloud.boot.openfeign.common;

import com.elitescloud.boot.CloudtSpringContextHolder;
import org.springframework.cloud.openfeign.FeignClientBuilder;
import org.springframework.lang.NonNull;
import org.springframework.util.StringUtils;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

/**
 * 动态调用客户端.
 *
 * @author Kaiser（wang shao）
 * @date 2022/8/1
 */
public class DynamicClientHelper {
    private static FeignClientBuilder feignClientBuilder;

    private static final Map<String, Object> CLIENT_CACHE = new ConcurrentHashMap<>();

    /**
     * 获取服务客户端
     *
     * @param serviceName 服务名称
     * @param clazz       服务接口类型
     * @param path        服务接口路径
     * @param <T>         接口类型
     * @return 服务接口实例
     */
    @SuppressWarnings("unchecked")
    public static <T> T getClient(@NonNull String serviceName, @NonNull Class<T> clazz, String path) {
        String key = serviceName + ":" + clazz.getName();
        Object client = CLIENT_CACHE.get(key);
        if (client != null) {
            return (T) client;
        }

        // 如果缓存没有，则构建
        synchronized (CLIENT_CACHE) {
            client = CLIENT_CACHE.get(key);
            if (client != null) {
                return (T) client;
            }

            FeignClientBuilder.Builder<T> clientBuilder = getFeignClientBuilder().forType(clazz, serviceName);
            if (StringUtils.hasText(path)) {
                clientBuilder.path(path);
            }
            client = clientBuilder.build();

            CLIENT_CACHE.put(key, client);
        }
        return (T) client;
    }

    /**
     * 构建客户端
     *
     * @param serviceName 服务名称
     * @param clazz 服务接口类型
     * @param path 服务接口路径
     * @param url 服务注册的url
     * @return 客户端service
     * @param <T> service类型
     */
    public static <T> T buildClient(@NonNull String serviceName, @NonNull Class<T> clazz, String path, String url) {
        FeignClientBuilder.Builder<T> clientBuilder = getFeignClientBuilder().forType(clazz, serviceName);
        if (StringUtils.hasText(path)) {
            clientBuilder.path(path);
        }
        if (StringUtils.hasText(url)) {
            clientBuilder.url(url);
        }
        return clientBuilder.build();
    }

    private static FeignClientBuilder getFeignClientBuilder() {
        if (feignClientBuilder == null) {
            var applicationContext = CloudtSpringContextHolder.getApplicationContext();
            feignClientBuilder = new FeignClientBuilder(applicationContext);
        }
        return feignClientBuilder;
    }
}
