package com.elitescloud.boot.openfeign.config;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.cloud.client.DefaultServiceInstance;
import org.springframework.cloud.client.ServiceInstance;
import org.springframework.cloud.client.loadbalancer.DefaultRequestContext;
import org.springframework.cloud.client.loadbalancer.Request;
import org.springframework.cloud.client.loadbalancer.RequestData;
import org.springframework.cloud.client.loadbalancer.reactive.ReactiveLoadBalancer;
import org.springframework.cloud.loadbalancer.blocking.client.BlockingLoadBalancerClient;

/**
 * 自定义负载均衡客户端.
 * <p>
 * 判断如果调用的服务和当前服务名一样，则直接调用自己
 *
 * @author Kaiser（wang shao）
 * @date 2022/9/21
 */
class CloudtBlockingLoadBalancerClient extends BlockingLoadBalancerClient {
    private static final Logger logger = LoggerFactory.getLogger(CloudtBlockingLoadBalancerClient.class);

    private final String applicationName;
    private final Integer port;
    private final ServiceInstance serviceInstance;

    @Override
    public <T> ServiceInstance choose(String serviceId, Request<T> request) {
        var instance = getServiceInstance(serviceId);
        if (instance != null) {
            logger.info("LoadBalancer ServiceInstance: {}, {}, {}", serviceId, instance.getHost(), instance.getPort());
            return instance;
        }

        instance = super.choose(serviceId, request);
        if (instance == null) {
            instance = this.attemptBuildServiceInstanceForIpAndPort(request.getContext());
        }
        logger.info("LoadBalancer ServiceInstance: {}, {}, {}", serviceId, instance == null ? null : instance.getHost(), instance == null ? null : instance.getPort());

        return instance;
    }

    public CloudtBlockingLoadBalancerClient(String applicationName, Integer port, ReactiveLoadBalancer.Factory<ServiceInstance> loadBalancerClientFactory) {
        super(loadBalancerClientFactory);
        this.applicationName = applicationName;
        this.port = port;
        this.serviceInstance = buildLocalInstance();
    }

    private <T> ServiceInstance attemptBuildServiceInstanceForIpAndPort(T context) {
        if (!(context instanceof DefaultRequestContext)) {
            return null;
        }
        var requestContext = (DefaultRequestContext) context;
        var clientRequest = requestContext.getClientRequest();
        if (!(clientRequest instanceof RequestData)) {
            return null;
        }
        var requestData = (RequestData)clientRequest;
        var url = requestData.getUrl();
        if (url == null) {
            return null;
        }

        return new DefaultServiceInstance(url.getAuthority(), url.getAuthority(), url.getHost(), url.getPort(), "https".equalsIgnoreCase(url.getScheme()));
    }

    private ServiceInstance getServiceInstance(String serviceId) {
        if (applicationName != null && applicationName.equals(serviceId)) {
            // 当前应用
            return serviceInstance;
        }

        return null;
    }

    private ServiceInstance buildLocalInstance() {
        return new DefaultServiceInstance(applicationName, applicationName, "127.0.0.1", port, false);
    }
}
