package com.elitescloud.boot.openfeign.config;

import cn.hutool.core.collection.CollUtil;
import com.elitescloud.boot.constant.ClassNameConstant;
import com.elitescloud.boot.openfeign.config.web.FeignClientRequestMappingHandlerMappingCustomizer;
import feign.Feign;
import feign.Response;
import feign.Target;
import feign.codec.ErrorDecoder;
import lombok.extern.log4j.Log4j2;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.boot.autoconfigure.AutoConfigureBefore;
import org.springframework.boot.autoconfigure.condition.ConditionalOnClass;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.cloud.client.loadbalancer.LoadBalancerClient;
import org.springframework.cloud.loadbalancer.config.BlockingLoadBalancerClientAutoConfiguration;
import org.springframework.cloud.loadbalancer.support.LoadBalancerClientFactory;
import org.springframework.cloud.openfeign.*;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.StringUtils;

import java.util.HashSet;
import java.util.Set;

/**
 * OpenFeign自动化配置.
 *
 * @author Kaiser（wang shao）
 * @date 2022/9/21
 */
@Configuration
@EnableConfigurationProperties(CloudtOpenFeignProperties.class)
@EnableFeignClients(basePackages = {"com.elitescloud", "com.elitesland"})
@AutoConfigureBefore({BlockingLoadBalancerClientAutoConfiguration.class, FeignAutoConfiguration.class})
@Log4j2
class CloudtOpenFeignAutoConfiguration {

    @Value("${spring.application.name:#{null}}")
    private String applicationName;
    @Value("${server.port:8080}")
    private Integer port;

    private final CloudtOpenFeignProperties properties;

    public CloudtOpenFeignAutoConfiguration(CloudtOpenFeignProperties properties) {
        this.properties = properties;
    }

    @Bean
    @ConditionalOnProperty(prefix = CloudtOpenFeignProperties.CONFIG_PREFIX, name = "direct-local", havingValue = "true", matchIfMissing = true)
    public CloudtTargeter cloudtTargeter() {
        var targeterProxy = new TargeterProxy(new Targeter() {
            @Override
            public <T> T target(FeignClientFactoryBean factory, Feign.Builder feign, FeignContext context, Target.HardCodedTarget<T> target) {
                // 参考 {@link org.springframework.cloud.openfeign.DefaultTargeter}
                return feign.target(target);
            }
        }, buildProxyServices(), properties);
        return new CloudtTargeter(targeterProxy);
    }

    /**
     * 异常打印
     *
     * @return
     */
    @Bean
    public ErrorDecoder errorDecoder() {
        return new ErrorDecoder.Default() {
            @Override
            public Exception decode(String methodKey, Response response) {
                var exp = super.decode(methodKey, response);
                log.error("OpenFeign处理异常：", exp);

                return exp;
            }
        };
    }

    /**
     * 处理webmvc的handler
     * <p>
     * 避免FeignClient的重复注册
     *
     * @return
     */
    @Bean
    public FeignClientRequestMappingHandlerMappingCustomizer feignClientRequestMappingHandlerMappingCustomizer() {
        return new FeignClientRequestMappingHandlerMappingCustomizer();
    }

    /**
     * 用户上下文信息传递interceptor
     *
     * @return
     */
    @Bean
    @ConditionalOnClass(name = ClassNameConstant.AUTHORIZATION_CLIENT)
    public FeignAuthenticationContextInterceptor feignAuthenticationContextInterceptor() {
        return new FeignAuthenticationContextInterceptor();
    }

    @Bean
    public CloudtHeaderInterceptor cloudtHeaderInterceptor() {
        return new CloudtHeaderInterceptor();
    }

    /**
     * traceId传递
     *
     * @return
     */
    @Bean
    public FeignTraceLogInterceptor feignLogTraceHandler() {
        return new FeignTraceLogInterceptor(applicationName);
    }

    /**
     * 自定义负载均衡
     *
     * @param loadBalancerClientFactory
     * @return
     */
    @Bean
    public LoadBalancerClient blockingLoadBalancerClient(LoadBalancerClientFactory loadBalancerClientFactory) {
        return new CloudtBlockingLoadBalancerClient(applicationName, port, loadBalancerClientFactory);
    }

    private Set<String> buildProxyServices() {
        Set<String> services = new HashSet<>();
        if (CollUtil.isNotEmpty(properties.getDirectLocalServices())) {
            // 是否代理所有
            boolean proxyAll = properties.getDirectLocalServices().stream().anyMatch("*"::equals);
            if (proxyAll) {
                services.add("*");
                return services;
            }

            services.addAll(properties.getDirectLocalServices());
        }

        // 代理当前应用
        if (StringUtils.hasText(applicationName)) {
            services.add(applicationName);
        }

        return services;
    }
}
